#include "PairAction.h"

int PairTable::read (char * filename)
{
  scalar q, error;
  FILE *fin;

  if ((fin = fopen (filename, "r")) == NULL)
    die ("Can't open file %s in Pairtable.\n", filename);

  for (int qindex = 1; qindex < NumqVals; qindex++)
    {
      fscanf (fin, " %lf %lf ", &q, &error);
      if (qindex == 1)
	qmin = q;
      if (qindex == (NumqVals - 1))
	qmax = q;
      for (int order =0; order < NumOrder; order++)
	fscanf(fin, " %lf ", &uk[qindex][order]);
    }
  //  printf ("qmin = %1.5f  qmax = %1.5f.\n", qmin, qmax);
  // Now extrapolate to q = 0
  for (int order=0; order < NumOrder; order++)
    uk[0][order] = 2.0 * uk[1][order] - uk[2][order];

  fclose (fin);
  // NOTE: NumqVals is actually 1 greater than the number of lines in
  // the data file.  We do this because we add one more value by
  // extrapolating to q <= 0.
  delta = (qmax - qmin) / (NumqVals -2);

  if (qmin > (delta * 1.00001))
    die ("The first value of q in the pairtable is > delta.\n");
  // Now adjust qmin for our extrapolation at the low values.  We
  // subtract off delta just in case the first value in the table does
  // not occur at q = delta.
  qmin -= delta;
}


scalar PotentialEnergy (Nvector R, PairAction PA)
{
  scalar V = 0.0;

  for (int e1=0; e1 < PA.NumElecs; e1++)
    {
      for (int e2=e1+1; e2 < PA.NumElecs; e2++)
	{
	  scalar dist = abs(R[e1] - R[e2]);
	  V += 1.0 / dist;
	}
      for (int ion=0; ion < PA.NumIons; ion++)
	{
	  scalar dist = abs(R[e1] - PA.Rions[ion]);
	  V -= PA.Zions[ion] / dist;
	}
    }
  return (V);
}

scalar PairAction::PrimitivePotentialLinkAction(Nvector R, Nvector Rprime)
{
  return (0.5 * tau * (PotentialEnergy (R, *this) +
		       PotentialEnergy(Rprime, *this)));
}



// Function PotentialLinkAction:  returns the potential part of the
// action for a single link.

scalar PairAction::PotentialLinkAction(Nvector R, Nvector Rprime)
{
  scalar EndAction = 0.0;  // End-point action
  scalar ODAction = 0.0;   // Action from off-diagonal terms

  // We'll need to loop over all of the interactions of the electrons
  // with the other electrons and with the ions

  for (int e1=0; e1 < NumElecs; e1++)
    {
      // First, we'll do the electron-electron terms
      for (int e2=e1+1; e2 < NumElecs; e2++)
	{

	  Vector<scalar> r(3);
	  Vector<scalar> rprime(3);
	  
	  r = R[e1] - R[e2];
	  rprime = Rprime[e1] - Rprime[e2];

	  scalar rmag = abs(r);
	  scalar rprimemag = abs(rprime);
	  scalar q = (rmag + rprimemag) / 2.0;

	  Vector<scalar> rdiff(3);
	  rdiff = r - rprime;
	  
	  scalar ssquared = dot_prod(rdiff, rdiff);
	  
	  // First, we do the end-point terms;
	  if (rmag < ee.qmax)
	    EndAction += 0.5 * ee.interp(rmag, 0);
	  else   // otherwise use semiclassical action
	    EndAction += 0.5 * tau / rmag;  

	  if(rprimemag < ee.qmax)
	    EndAction += 0.5 * ee.interp(rprimemag, 0);
	  else   // otherwise use semiclassical action
	    EndAction += 0.5 * tau / rprimemag;


	  scalar s2order = ssquared;  // s^(2*order)
	  
	  // Now we loop over the remaining orders.  Note that since
	  // we are using only Coulomb potentials, we don't have to
	  // worry about j dependence.
	  if (q < ee.qmax)
	    {
	      for (int order=1; order < ee.NumOrder; order++)
		{
		  ODAction += ee.interp(q, order) * s2order;
		  s2order *= ssquared;
		}
	    }
	}
      // Now we must do the electron-ion terms
      for (int ion=0; ion < NumIons; ion++)
	{

	  Vector<scalar> r(3);
	  Vector<scalar> rprime(3);
	  
	  r = R[e1] - Rions[ion];
	  rprime = Rprime[e1] - Rions[ion];

	  scalar rmag = abs(r);
	  scalar rprimemag = abs(rprime);
	  scalar q = (rmag + rprimemag) / 2.0;

	  Vector<scalar> rdiff(3);
	  rdiff = r - rprime;
	  
	  scalar ssquared = dot_prod(rdiff, rdiff);
	  
	  // First, we do the end-point terms;
	  if (rmag < eion[Species[ion]].qmax)
	    EndAction += 0.5 * eion[Species[ion]].interp(rmag, 0);
	  else   // otherwise use semiclassical action
	    EndAction -= 0.5 * Zions[ion] * tau / rmag;  

	  if(rprimemag < eion[Species[ion]].qmax)
	    EndAction += 0.5 * eion[Species[ion]].interp(rprimemag, 0);
	  else   // otherwise use semiclassical action
	    EndAction -= 0.5 * tau / rprimemag;


	  scalar s2order = ssquared;  // s^(2*order)
	  
	  // Now we loop over the remaining orders.  Note that since
	  // we are using only Coulomb potentials, we don't have to
	  // worry about j dependence.
	  if (q < eion[Species[ion]].qmax)
	    {
	      for (int order=1; order < eion[Species[ion]].NumOrder; order++)
		{
		  ODAction += eion[Species[ion]].interp(q, order) * s2order;
		  s2order *= ssquared;
		}
	    }
	}
    }
  return (EndAction + ODAction);
}
  

scalar PairAction::KineticLinkAction(Nvector R, Nvector Rprime)
{
  scalar Kaction = 0.0;
  Vector<scalar> rdiff(3);
  
  // Loop over the electrons and sum up the spring terms
  for (int i=0; i<NumElecs; i++)
    {
      rdiff = R[i] - Rprime[i];
      Kaction += dot_prod(rdiff, rdiff);
    }

  // Now multiply by the appropriate constant factor
  Kaction *= 0.25 / (D * tau);
  return (Kaction);
}


// The PairAction::read function will read a file containing all the
// parameters for the pair action.  It has the following fixed format:
//
// tau
// Number of Electrons
// Number of Ions
// Number of Ionic Species
// e-e PairTable   filename
// e-e GridType
// e-e NumqVals   NumOrder
// Species#1 PairTable filename
// Species#1 PairTable GridType
// Species#1 NumqVals   NumOrder
// Species#2 PairTable filename
// Species#2 PairTable GridType
// Species#2 NumqVals   NumOrder
// ..............
// Ion#1 species  charge   x   y  z
// Ion#2 species  charge   x   y  z
// Ion#3 species  charge   x   y  z
// ..............

void
PairAction::read(char *filename)
{
  // First, open the file
  FILE *fin;
  if ((fin = fopen (filename, "r")) == NULL)
    die ("Can't open PairAction file %s.\n", filename);
  
  // Read in tau
  fscanf (fin, " %lf ", &tau);
  fscanf (fin, " %d ", &NumElecs);
  fscanf (fin, " %d ", &NumIons);
  fscanf (fin, " %d ", &NumSpecies);

  char PairTableName[500];
  char GridTypeName[500];
  
  // First, read in the e-e table
  fgets (PairTableName, 500, fin);
  fgets (GridTypeName, 500, fin);
  PairTableName[strlen(PairTableName)-1] = '\0';
  GridTypeName[strlen(GridTypeName)-1] = '\0';
  if(!strcmp(GridTypeName, "Linear"))
    ee.GridType = Linear;
  else if (!strcmp(GridTypeName, "Log"))
    ee.GridType = Log;
  else
    die ("Unrecognized Grid Type in PairAction::read.\n");  

  fscanf (fin, " %d ", &ee.NumqVals);
  // Add 1 to the number of q vals to hold extrapolation to zero 
  ee.NumqVals++;
  fscanf (fin, " %d ", &ee.NumOrder);

  // Now initialize the table
  ee.uk.init(ee.NumqVals, ee.NumOrder);
  // Finally, read in the pair table from disk
  ee.read (PairTableName);

  
  
  // First inintialize the vector of PairTables:
  eion.newsize(NumSpecies);
  // Now we loop over the ionic species, reading in the pair tables 
  for (int species=0; species<NumSpecies; species++)
    {
      fgets (PairTableName, 500, fin);
      fgets (GridTypeName, 500, fin);
      PairTableName[strlen(PairTableName)-1] = '\0';
      GridTypeName[strlen(GridTypeName)-1] = '\0';
      
      if(!strcmp(GridTypeName, "Linear"))
	eion[species].GridType = Linear;
      else if (!strcmp(GridTypeName, "Log"))
	eion[species].GridType = Log;
      else
	die ("Unrecognized Grid Type in PairAction::read.\n");  
      
      fscanf (fin, " %d ", &eion[species].NumqVals);
      // Add 1 to the number of q vals to hold extrapolation to zero 
      eion[species].NumqVals++;
      fscanf (fin, " %d ", &eion[species].NumOrder);
      
      // Now initialize the table
      eion[species].uk.init(eion[species].NumqVals, eion[species].NumOrder);
      // Finally, read in the pair table from disk
      eion[species].read (PairTableName);
    }

  // Now we have to read in the rest of the info.
  // Let's initialize the vectors to hold the information about the
  // ions:
  Species.newsize(NumIons);
  Zions.newsize(NumIons);
  Rions.init(NumIons, 3);
  for (int ion=0; ion < NumIons; ion++)
    {
      fscanf (fin, " %d ", &Species[ion]);
      fscanf (fin, " %lf ", &Zions[ion]);
      fscanf (fin, " %lf %lf %lf ",
	      &Rions[ion][0], &Rions[ion][1], &Rions[ion][2]);
    }

  // The only thing left to do is close the file.
  fclose (fin);
}








