/****************************************************************************
 * Copyright (C) 1999 Matthew Frank
 *
 *                        All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the author's
 * name not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission.
 *  
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Matt Frank, MIT Laboratory for Computer Science,
 *         mfrank@lcs.mit.edu
 *         1999-Jul-16
 *
 ***************************************************************************/

#ifndef __IEEE_FP_H
#define __IEEE_FP_H

#include <stddef.h>
#include <machine/inttypes.h>

/****************************************************************************
 * IEEE 754 single precision floating point format:
 *   31 30 x x x x x x 23 22 x x x x x x x x x x x x x x x x x x x x x  0
 * | s |      exp        |                mantissa                       |
 *
 * s = 0 => positive, s = 1 => neg
 * exp is a biased signed integer in the range [-127, 128]
 * mantissa stores the bottom 23 bits of the fraction.
 *
 * numbers with exponents in the range [-126,127] are normalized  (the high bit is
 *   implicitly 1).  The number represented by a particular bit pattern is
 *         (-1)^s (1 + (mantissa / 2^23)) 2^(exp-127)
 * numbers with exponent 128 are special:
 *   if the mantissa is 0 then they represent + or - Infinity (overflow or div by 0)
 *   if the mantissa is non-zero then they represent NaN (e.g. sqrt(-1)).
 * numbers with exponent -127 are special:
 *   if the mantissa is 0 then they represent + or - 0.
 *   if the mantissa is non-zero then they represent denormalized numbers:
 *       (-1)^s (mantissa / 2^23) 2^(-127)
 * 
 ***************************************************************************/

typedef union
__ieee754_convert_union
{
  float value;
  __uint32_t word;
} __ieee754_float_int_union;
 
#define IEEE754_FLOAT_BIAS	0x7f /* Added to exponent.  */

static inline
float
_cvt_int2float(__uint32_t u)
{
  __ieee754_float_int_union fiu;
  fiu.word = u;
  return fiu.value;
}

static inline
__uint32_t
_cvt_float2int(float f)
{
  __ieee754_float_int_union fiu;
  fiu.value = f;
  return fiu.word;
} 

#define NAN _cvt_int2float(0x7fc00000UL)
#define HUGE_VAL _cvt_int2float(0x7f800000UL)

static inline
int
_ieee754_get_sign(__uint32_t f)
{
  return f >> 31;
}

static inline
int
_ieee754_get_exponent(__uint32_t f)
{
  return ((f >> 23) & 0xff) - IEEE754_FLOAT_BIAS;
}

static inline
int
_ieee754_get_mantissa(__uint32_t f)
{
  const __uint32_t mantissa_bits = 23;
  const __uint32_t hidden_bit = (1 << mantissa_bits);
  const __uint32_t mantissa_mask = (hidden_bit - 1);

  return (f & mantissa_mask) + hidden_bit;
}

static inline
__uint32_t
_ieee754_make_float(__uint32_t sign, __uint32_t exponent, __uint32_t mantissa)
{
  const __uint32_t mantissa_bits = 23;
  const __uint32_t mantissa_mask = ((1 << mantissa_bits) - 1);
  const __uint32_t exponent_bits = 8;
  const __uint32_t exponent_mask = ((1 << exponent_bits) - 1);
  __uint32_t ix =
    (sign << (mantissa_bits + exponent_bits)) |
    (((exponent + IEEE754_FLOAT_BIAS) & exponent_mask) << mantissa_bits) | 
    (mantissa & mantissa_mask);

  return ix;
}

/* divide a normalized floating point number by 2.0.  Does not work
   for subnormals */
static inline
float
_ieee754_half(float x)
{
  __uint32_t ix = _cvt_float2int(x);
  __uint32_t ix_over_2 = (ix - (1 << 23)); /* subtract 1 from exponent */
  return _cvt_int2float(ix_over_2);
}

#endif  /* __IEEE_FP_H */
