/*
 *      Command line frontend program
 *
 *      Copyright (c) 1999 Mark Taylor
 *                    2000 Takehiro TOMIANGA
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* $Id: main.c,v 1.1.1.1 2005/08/10 15:16:28 mfrank Exp $ */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <assert.h>
#include <stdio.h>

#ifdef STDC_HEADERS
# include <stdlib.h>
# include <string.h>
#else
# ifndef HAVE_STRCHR
#  define strchr index
#  define strrchr rindex
# endif
char   *strchr(), *strrchr();
# ifndef HAVE_MEMCPY
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#  define memmove(d, s, n) bcopy ((s), (d), (n))
# endif
#endif

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#ifdef __sun__
/* woraround for SunOS 4.x, it has SEEK_* defined here */
#include <unistd.h>
#endif

/*
 main.c is example code for how to use libmp3lame.a.  To use this library,
 you only need the library and lame.h.  All other .h files are private
 to the library.
*/
#include "lame.h"

#include "main.h"
#include "get_audio.h"
#include "portableio.h"

/* PLL 14/04/2000 */
#if macintosh
#include <console.h>
#endif

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif



/* GLOBAL VARIABLES.  set by parse_args() */
/* we need to clean this up */
sound_file_format input_format = sf_wave;   
int swapbytes = 0;              /* force byte swapping   default=0*/
int silent = 0;                 /* Verbosity */
int ignore_tag_errors = 0;      /* Ignore errors in values passed for tags */
int brhist = 1;
float update_interval = 0.0;      /* to use Frank's time status display */
int mp3_delay = 0;              /* to adjust the number of samples truncated
                               during decode */
int mp3_delay_set = 0;          /* user specified the value of the mp3 encoder 
                               delay to assume for decoding */

int enc_delay = -1;
int enc_padding = -1;
int disable_wav_header = 0;
mp3data_struct mp3input_data; /* used by MP3 decoder (disabled) */
int print_clipping_info = 0;      /* print info whether waveform clips */

int in_signed=1;
int in_unsigned=0;
int in_endian=order_littleEndian;
int in_bitwidth=16;

/************************************************************************
*
* main
*
* PURPOSE:  MPEG-1,2 Layer III encoder with GPSYCHO
* psychoacoustic model.
*
************************************************************************/


FILE   *
init_files(lame_global_flags * gf, char *inPath, char *outPath)
{
    FILE   *outf;
    /* Mostly it is not useful to use the same input and output name.
       This test is very easy and buggy and don't recognize different names
       assigning the same file
     */
    if (0 != strcmp("-", outPath) && 0 == strcmp(inPath, outPath)) {
        fprintf(stderr, "Input file and Output file are the same. Abort.\n");
        return NULL;
    }

    /* open the wav/aiff/raw pcm or mp3 input file.  This call will
     * open the file, try to parse the headers and
     * set gf.samplerate, gf.num_channels, gf.num_samples.
     * if you want to do your own file input, skip this call and set
     * samplerate, num_channels and num_samples yourself.
     */
    init_infile(gf, inPath);
    if ((outf = init_outfile(outPath, lame_get_decode_only(gf))) == NULL) {
        fprintf(stderr, "Can't init outfile '%s'\n", outPath);
        return NULL;
    }

    return outf;
}

int
lame_encoder(lame_global_flags * gf, FILE * outf, int nogap, char *inPath,
             char *outPath)
{
    unsigned char mp3buffer[LAME_MAXMP3BUFFER];
    int     Buffer[2][1152];
    int     iread, imp3;
    static const char *mode_names[2][4] = {
        {"stereo", "j-stereo", "dual-ch", "single-ch"},
        {"stereo", "force-ms", "dual-ch", "single-ch"}
    };
    int     frames;

    if (silent < 10) {
        lame_print_config(gf); /* print useful information about options being used */

        fprintf(stdout, "Encoding %s%s to %s\n",
                strcmp(inPath, "-") ? inPath : "<stdin>",
                strlen(inPath) + strlen(outPath) < 66 ? "" : "\n     ",
                strcmp(outPath, "-") ? outPath : "<stdout>");

        fprintf(stdout,
                "Encoding as %g kHz ", 1.e-3 * lame_get_out_samplerate(gf));

	{
            const char *appendix = "";

            switch (lame_get_VBR(gf)) {
            case vbr_mt:
            case vbr_rh:
            case vbr_mtrh:
                appendix = "ca. ";
                fprintf(stderr, "VBR(q=%i)", lame_get_VBR_q(gf));
                break;
            case vbr_abr:
                fprintf(stderr, "average %d kbps",
                        lame_get_VBR_mean_bitrate_kbps(gf));
                break;
            default:
                fprintf(stderr, "%3d kbps", lame_get_brate(gf));
                break;
            }
            fprintf(stderr, " %s MPEG-%u%s Layer III (%s%gx) qval=%i\n",
                    mode_names[lame_get_force_ms(gf)][lame_get_mode(gf)],
                    2 - lame_get_version(gf),
                    lame_get_out_samplerate(gf) < 16000 ? ".5" : "",
                    appendix,
                    0.1 * (int) (10. * lame_get_compression_ratio(gf) + 0.5),
                    lame_get_quality(gf));
        }

        if (silent <= -10)
            lame_print_internals(gf);

        fflush(stderr);
    }

    /* encode until we hit eof */
    do {
        /* read in 'iread' samples */
        iread = get_audio(gf, Buffer);
        frames = lame_get_frameNum(gf);

        printf("%d\n", frames);

        /* encode */
        imp3 = lame_encode_buffer_int(gf, Buffer[0], Buffer[1], iread,
                                      mp3buffer, sizeof(mp3buffer));

        /* was our output buffer big enough? */
        if (imp3 < 0) {
            if (imp3 == -1)
                fprintf(stderr, "mp3 buffer is not big enough... \n");
            else
                fprintf(stderr, "mp3 internal error:  error code=%i\n", imp3);
            return 1;
        }

        if (fwrite(mp3buffer, 1, imp3, outf) != imp3) {
            fprintf(stderr, "Error writing mp3 output \n");
            return 1;
        }

    } while (iread);

    if (nogap) 
        imp3 = lame_encode_flush_nogap(gf, mp3buffer, sizeof(mp3buffer)); /* may return one more mp3 frame */
    else
        imp3 = lame_encode_flush(gf, mp3buffer, sizeof(mp3buffer)); /* may return one more mp3 frame */

    if (imp3 < 0) {
        if (imp3 == -1)
            fprintf(stderr, "mp3 buffer is not big enough... \n");
        else
            fprintf(stderr, "mp3 internal error:  error code=%i\n", imp3);
        return 1;

    }

    fwrite(mp3buffer, 1, imp3, outf);

    return 0;
}






void
brhist_init_package(lame_global_flags * gf)
{
#ifdef BRHIST
    if (brhist) {
        if (brhist_init
            (gf, lame_get_VBR_min_bitrate_kbps(gf),
             lame_get_VBR_max_bitrate_kbps(gf))) {
            /* fail to initialize */
            brhist = 0;
        }
    }
    else {
        brhist_init(gf, 128, 128); /* Dirty hack */
    }
#endif
}


  

void print_lame_tag_leading_info(lame_global_flags *gf) {
    if(lame_get_bWriteVbrTag(gf))
      printf("Writing LAME Tag...");
}

void print_trailing_info(lame_global_flags *gf)
{
    if (lame_get_bWriteVbrTag(gf))
      printf("done\n");

    if (lame_get_findReplayGain(gf) ) {
      int RadioGain = lame_get_RadioGain(gf);
      printf("ReplayGain: %s%.1fdB\n", RadioGain > 0 ? "+" : "", ((float)RadioGain) / 10.0);
      if (RadioGain > 0x1FE || RadioGain < -0x1FE) 
	printf("WARNING: ReplayGain exceeds the -51dB to +51dB range. Such a result is too\n"
	       "         high to be stored in the header.\n" );
    }
	
    /* if (the user requested printing info about clipping) and (decoding
       on the fly has actually been performed) */
    if(print_clipping_info && lame_get_decode_on_the_fly(gf)) {
      float noclipGainChange = (float)lame_get_noclipGainChange(gf) / 10.0;
      float noclipScale = lame_get_noclipScale(gf);

      if (noclipGainChange > 0.0) { /* clipping occurs */
	printf("WARNING: clipping occurs at the current gain. Set your decoder to decrease\n"
	       "         the  gain  by  at least %.1fdB or encode again ", noclipGainChange); 
                                                                            
	/* advice the user on the scale factor */
	if (noclipScale > 0) {
	  printf("using  --scale %.2f\n", noclipScale);
	  printf("         or less (the value under --scale is approximate).\n" );
	}
	else {
          /* the user specified his own scale factor. We could suggest 
           * the scale factor of (32767.0/gfp->PeakSample)*(gfp->scale)
           * but it's usually very inaccurate. So we'd rather advice him to 
           * disable scaling first and see our suggestion on the scale factor then. */
	  printf("using --scale <arg>\n"
	         "         (For   a   suggestion  on  the  optimal  value  of  <arg>  encode\n"
	         "         with  --scale 1  first)\n" );
	}
		    
      }
      else { /* no clipping */
	if (noclipGainChange > -0.1)
	  printf("\nThe waveform does not clip and is less than 0.1dB away from full scale.\n" );
	else                                                                                        
	  printf("\nThe waveform does not clip and is at least %.1fdB away from full scale.\n", -noclipGainChange);
      }                                                                                             
    }
  
}


double foobar = 0.06;
float exebar = 0.06f;

int
#if defined(__polyflow__)
_user_main(int argc, char **argv)
#else
main(int argc, char **argv)
#endif
{
    int     ret;
    lame_global_flags *gf;
    char    outPath[PATH_MAX + 1];
    char    nogapdir[PATH_MAX + 1];
    char    inPath[PATH_MAX + 1];

    /* support for "nogap" encoding of up to 200 .wav files */
#define MAX_NOGAP 200
    int    nogapout = 0;
    int     max_nogap = MAX_NOGAP;
    char   *nogap_inPath[MAX_NOGAP];

    int     i;
    FILE   *outf;

#if macintosh
    argc = ccommand(&argv);
#endif

#if defined(_WIN32)
   /* set affinity back to all CPUs.  Fix for EAC/lame on SMP systems from
     "Todd Richmond" <todd.richmond@openwave.com> */
    typedef BOOL (WINAPI *SPAMFunc)(HANDLE, DWORD);
    SPAMFunc func;
    SYSTEM_INFO si;

    if ((func = (SPAMFunc)GetProcAddress(GetModuleHandle("KERNEL32.DLL"),
        "SetProcessAffinityMask")) != NULL) {
        GetSystemInfo(&si);
        func(GetCurrentProcess(), si.dwActiveProcessorMask);
    }
#endif


#ifdef __EMX__
    /* This gives wildcard expansion on Non-POSIX shells with OS/2 */
    _wildcard(&argc, &argv);
#endif

    for (i = 0; i < max_nogap; ++i) {
        nogap_inPath[i] = malloc(PATH_MAX + 1);
    }

    memset(inPath, 0, sizeof(inPath));

    /* initialize libmp3lame */
    if (NULL == (gf = lame_init())) {
        fprintf(stderr, "fatal error during initialization\n");
        return 1;
    }
    if (argc != 3) {
        printf("usage: %s <infile>.wav <outfile>.mp3\n", argv[0]);
        return 1;
    }

    /* initialize to "standard" preset: */
    ret = 0;
    id3tag_init (gf);
    lame_set_preset(gf, STANDARD);

    memset(inPath, 0, PATH_MAX+1);
    memset(outPath, 0, PATH_MAX+1);

    strncpy(inPath, argv[1], PATH_MAX+1);
    strncpy(outPath, argv[2], PATH_MAX+1);

    lame_set_findReplayGain(gf,1);

    if (ret < 0)
        return ret == -2 ? 0 : 1;

    if (update_interval < 0.)
        update_interval = 2.;

    if (outPath[0] != '\0' && max_nogap>0) {
        strncpy(nogapdir, outPath, PATH_MAX + 1);  
        nogapout = 1;
    }

    /* initialize input file.  This also sets samplerate and as much
       other data on the input file as available in the headers */
    if ((outf = init_files(gf, inPath, outPath)) == NULL) {
        return -1;
    }

    /* Now that all the options are set, lame needs to analyze them and
     * set some more internal options and check for problems
     */
    i = lame_init_params(gf);
    if (i < 0) {
        fprintf(stderr, "fatal error during initialization\n");
        return i;
    }

    if (silent > 0 || lame_get_VBR(gf) == vbr_off) {
        brhist = 0;     /* turn off VBR histogram */
    }

    /*
     * encode a single input file
     */
    brhist_init_package(gf);
    ret = lame_encoder(gf, outf, 0, inPath, outPath);
            
    if (silent<=0) print_lame_tag_leading_info(gf);
    lame_mp3_tags_fid(gf, outf); /* add VBR tags to mp3 file */
	    
    if (silent<=0) print_trailing_info(gf);
            
    fclose(outf); /* close the output file */
    close_infile(); /* close the input file */
    lame_close(gf);

    return ret;
}



