#include "compress.h"

/*
 * Decompress stdin to stdout.  This routine adapts to the codes in the
 * file building the "string" table on-the-fly; requiring no table to
 * be stored in the compressed file.  The tables used herein are shared
 * with those of the compress() routine.  See the definitions above.
 */

static unsigned short tab_prefix[1<<BITS];
static char_type tab_suffix[1<<BITS];

void
decompress(const int maxbits,
           const int block_compress,
           stream_state *in_stream,
           stream_state *out_stream)
{
  code_int incode, oldcode;
  int finchar;
  code_int free_ent = FIRST(block_compress);
  input_buf_t input_buf;

  input_initialize(&input_buf, in_stream);
  /*
   * initialize the first 256 entries in the table.
   */
  for ( incode = 0; incode < 256; incode++) {
    tab_prefix[incode] = 0;
    tab_suffix[incode] = (char_type)incode;
  }

  oldcode = getcode(&input_buf);
  finchar = oldcode;
  if(oldcode == EOF)            /* EOF already? */
    return;			/* Get out of here */
  /* first code must be 8 bits = char */
  putbyte(out_stream, (char_type)oldcode);

  while ( (incode = getcode(&input_buf)) > EOF ) {
    int stackp = 0;
    code_int code;
    char_type the_stack[1<<BITS];

    if ( (incode == CLEAR) && block_compress ) {
      free_ent = FIRST(block_compress) - 1;
      if (input_clear_codes(&input_buf) == EOF)
	break;      /* untimely death! */
      incode = getcode(&input_buf);
    }

    code = incode;

    /* Special case for KwKwK string. */
    if ( code >= free_ent ) {
      the_stack[stackp++] = finchar;
      code = oldcode;
    }

    /* Generate output characters in reverse order */
    while ( code >= 256 ) {
      the_stack[stackp++] = tab_suffix[code];
      code = tab_prefix[code];
    }
    the_stack[stackp++] = tab_suffix[code];

    /* And put them out in forward order */
    do
      putbyte ( out_stream, the_stack[--stackp] );
    while ( stackp > 0 );

    /* Generate the new entry. */
    if ( free_ent < MAX_MAX_CODE(maxbits) ) {
      tab_prefix[free_ent] = (unsigned short)oldcode;
      tab_suffix[free_ent] = tab_suffix[code];
      free_ent = free_ent + 1;
      if (input_check_bits(free_ent, maxbits, &input_buf) == EOF) {
	break;			/* untimely death! */
      }
    }
    /* Remember previous codes. */
    finchar = tab_suffix[code];
    oldcode = incode;
  }
}
