#include "sparse_memory.h"

using namespace std;

void
sparse_memory::print_table()
{
  for (unsigned i = 0; i < TABLE_SIZE; i++) {
    if (hashtab[i].tag != EMPTY) {
      printf("%x: %x -> %x\n", i, hashtab[i].tag * PAGE_SIZE, hashtab[i].paddr);
    }
  }
}

void
sparse_memory::copy_linked_page(int32_t x_key, int32_t x_link_key) {
  //if(debug_mask) printf("sparse_memory:copy_linked_page key %d link_key %d\n", x_key, x_link_key);
  assert(linked_mem != NULL);
  hashtab[x_key].tag = linked_mem->hashtab[x_link_key].tag;
  hashtab[x_key].paddr = (uint32_t) (new char[PAGE_SIZE]);

  if (hashtab[x_key].paddr == 0) {
    fprintf(stderr, "OUT OF MEMORY IN imem::copy_page()!\n");
    abort();
  }

  for(uint32_t p = 0; p < PAGE_SIZE; ++p)
    *((char *)hashtab[x_key].paddr + p) =  *((char *)linked_mem->hashtab[x_link_key].paddr + p);
}

void
sparse_memory::compare_and_sync_with_linked() {
  //if(debug_mask) printf("sparse_memory::compare_and_sync_with_linked\n");
  assert(linked_mem != NULL);

  uint32_t removed = 0;
  for (uint32_t i=0; i<TABLE_SIZE; ++i) {
    if (hashtab[i].tag != EMPTY) {
      int32_t link_key;
      if(linked_mem->find_key(hashtab[i].tag, link_key)) {
        // compare page data
        bool page_match = true;
        for(uint32_t p = 0; p < PAGE_SIZE; ++p) {
          if(*((char *)hashtab[i].paddr + p) !=  
             *((char *)linked_mem->hashtab[link_key].paddr + p)) {
            page_match = false;
            break;
          }
        }
        if(page_match) {
          delete [] (char *) hashtab[i].paddr;
          hashtab[i].tag = EMPTY;
          hashtab[i].paddr = 0;
          ++removed;
        }
      }
    }
  }
  page_count -= removed;
}

void 
sparse_memory::sync_identical_with_linked() {
  assert(linked_mem != NULL);
  
  if(page_count > linked_mem->get_page_count() / 8) {
    compare_and_sync_with_linked();
  }
}

void
sparse_memory::sync_with_linked() {
  //if(debug_mask) printf("sparse_memory::sync_with_linked\n");
  assert(linked_mem != NULL);

  for (uint32_t i=0; i<TABLE_SIZE; ++i) {
    if (hashtab[i].tag != EMPTY) {
      delete [] (char *) hashtab[i].paddr;
      hashtab[i].tag = EMPTY;
      hashtab[i].paddr = 0;
    }
    else
      assert(hashtab[i].paddr == 0);
  }
  page_count = 0;
}

// These are all slow, but they should be correct, and they aren't on
// any critical paths.  Because we are copying between two different
// memories we don't need to worry about overlap.
void
sparse_memory::memcpy_to_host(void* dest, const uint32_t src, size_t n)
{
  //if(debug_mask) printf("sparse_memory::memcpy_to_host\n");
  char* output = (char*)dest;
  for (size_t i = 0; i < n; i++) {
    output[i] = load_uint8(src + i);
  }
}

void
sparse_memory::memcpy_from_host(uint32_t dest, const void* src, size_t n)
{
  //if(debug_mask) printf("sparse_memory::memcpy_from_host\n");
  char* input = (char*)src;
  for (size_t i = 0; i < n; i++) {
    store_uint8(input[i], dest + i);
  }
}

void
sparse_memory::strcpy_to_host(char* dest, const uint32_t src)
{
  //if(debug_mask) printf("sparse_memory::strcpy_to_host\n");
  uint8_t data;
  uint32_t s = src;
  do {
    data = load_uint8(s++);
    *dest = data;
    dest++;
  } while (data != 0);
  
}

void
sparse_memory::strcpy_from_host(uint32_t dest, const char* src)
{
  //if(debug_mask) printf("sparse_memory::strcpy_from_host\n");
  uint8_t data;
  do {
    data = *src;
    store_uint8(data, dest++);
    src++;
  } while (data != 0);
}

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

void
sparse_memory::emulate_syscall(uint32_t cmd_addr)
{
  //if(debug_mask) printf("sparse_memory::emulate_syscall\n");
  uint32_t dispatch_code = load_uint32(cmd_addr);
  uint32_t arg1 = load_uint32(cmd_addr + 8);
  uint32_t arg2 = load_uint32(cmd_addr + 12);
  uint32_t arg3 = load_uint32(cmd_addr + 16);
  uint32_t result = (uint32_t)-1;

  fflush(stdout);
  fflush(stderr);

  switch (dispatch_code) {
  case 0x80: {
    char str[4096];
    int flags = 0;
    strcpy_to_host(str, arg1);
    if (arg2 & 0x1) {
      flags |= O_WRONLY;
    }
    if (arg2 & 0x0200) {
      flags |= O_CREAT;
    }
    if (arg2 & 0x0400) {
      flags |= O_TRUNC;
    }
    result = open(str, flags, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
    break;
  }
  case 0x81: {
    result = close(arg1);
    break;
  }
  case 0x82: {
    char buf[arg3];
    result = read(arg1, buf, arg3);
    memcpy_from_host(arg2, buf, arg3);
    break;
  }
  case 0x83: {
    char buf[arg3];
    memcpy_to_host(buf, arg2, arg3);
    result = write(arg1, buf, arg3);
    break;
  }
  case 0x84: {
    result = lseek(arg1, arg2, arg3);
    break;
  }
  case 0x85: {
    char str[4096];
    strcpy_to_host(str, arg1);
    result = unlink(str);
    break;
  }
  case 0x86: {
    char str1[4096];
    char str2[4096];
    strcpy_to_host(str1, arg1);
    strcpy_to_host(str2, arg2);
    result = link(str1, str2);
    break;
  }
  case 0x87: {
    char str[4096];
    struct stat stat_obj;
    strcpy_to_host(str, arg1);
    result = stat(str, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x88: {
    struct stat stat_obj;
    result = fstat(arg1, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x89: {
    char str[4096];
    struct stat stat_obj;
    strcpy_to_host(str, arg1);
    result = lstat(str, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x90: {
    char str[4096];
    strcpy_to_host(str, arg1);
    result = chmod(str, arg2);
    break;
  }
  case 0x91: {
    char str1[4096];
    char str2[4096];
    strcpy_to_host(str1, arg1);
    strcpy_to_host(str2, arg2);
    result = rename(str1, str2);
    break;
  }
  case 0x92: {
    char name_template[4096];
    strcpy_to_host(name_template, arg1);
    result = mkstemp(name_template);
    break;
  }
  case 0x93: {
    result =  ftruncate(arg1, arg2);
    printf("Called the real ftruncate\n");
    break;
  }

  default: 	{
    fprintf(stderr, "bogus unix syscall %x\n", dispatch_code);
  }
  }
  store_uint32(result, cmd_addr + 4);
}
