#ifndef SCHEDULER_H
#define SCHEDULER_H

#include <vector>
#include <deque>
#include <list>
#include <map>
#include <algorithm>
#include <set>
#include "circuit.h"
#include "regfile_circuit.h"
#include "decode.h"
#include "pretty_print.h"
#include "issue_buffer.h"
#include "dependency_window.h"
#include "exec_unit.h"
#include "globals.h"
#include <ext/hash_map>

using namespace std;

// Global variable potentially defined on command line
extern uint32_t num_phys_regs;
extern uint32_t issue_buffer_size;
extern uint32_t num_contexts;
extern uint32_t debug_mask;            // scoreboard stage debug_mask is 0x00000008

extern uint64_t lsq_store_stall_cnt;
extern uint64_t lsq_load_stall_cnt;

extern uint64_t cycle_count;

extern uint32_t superpipeline_factor;
extern bool     schd_reg_dep_first;
extern bool     schd_histogram_accounting;
extern bool     bullet_proof_check_cycle;

class issue_buffer_object {
public:
  decoder::instr_h instr;
  bool issued;
  bool completed;
  
  issue_buffer_object(decoder::instr_h x_instr = decoder::noop) : 
    instr(x_instr),
    issued(false),
    completed(false)
  { }
};

class scheduler : circuit
{
  typedef pair<uint32_t, uint32_t> ready_reg_t;
  typedef map<uint64_t, uint64_t> cancel_range_t;
  typedef deque<ready_reg_t> reg_list_t;
  typedef issue_buffer_c<issue_buffer_object> issue_buffer_t;
  typedef map<uint32_t, uint32_t> histogram_map_t;

 private:
  execution_unit* the_exec_unit; // for allocating slots in the store buffer at dispatch time
  dependency_window_c dependency_window;
  issue_buffer_t issue_buffer;

  vector<uint8_t> reg_avail;
  reg_list_t ready_registers;

  // static members
  static vector<cancel_range_t> cancel_ranges; // inside_cancel_ranges() is static

  // histogram data gathering
  vector<uint32_t> instr_dep_count;
  vector<histogram_map_t> histogram_deps;
  vector<uint32_t> dispatch_dep_count;
  histogram_map_t histogram_writeback_chain_length;
  histogram_map_t histogram_dependency_window;
  histogram_map_t histogram_issue_buffer;
  histogram_map_t histogram_wakeup_bandwidth;
  uint32_t cycle_wakeup_success;
  histogram_map_t histogram_wakeup_sets;

  //stats
  uint32_t wakeup_attempts_total;
  uint32_t wakeup_attempts_success;
  uint32_t wakeup_attempts_instr_not_ready;
  uint32_t wakeup_attempts_wasted;
  uint64_t scheduler_size_sum;

  // scheduler instruction handleing
  bool safe_instr(const decoder::instr_h&) const; //checks if an instruction is a syscall or not (if syscall, better be at head of ROB)
  bool registers_ready(const decoder::instr_h&) const;
  bool ready_to_issue(size_t);
  static bool inside_cancel_range(const uint32_t, const uint64_t);
  void set_cancel_ts(const uint32_t, const uint64_t, const uint64_t);

  void issue_buffer_insert(decoder::instr_h);
  void register_ready_insert(const ready_reg_t&);
  void issue_instr(const size_t slot);
  void cancel_instr(const decoder::instr_h& instr_h);
  void cancel_slot(const size_t slot);
  void complete_slot(const size_t slot);

  // scheduler core
  void handle_statistics();
  void handle_misspeculation();
  void handle_writeback();
  void handle_nrms();
  void wakeup();
  void dispatch();
  void issue();
  void handle_retirement();
  void set_stall_conditions();
    
  void recalc();

public:
  void account_phys_regs(vector<uint32_t>&) const;

  ~scheduler() {
  }

  scheduler(execution_unit* exec_unit) :
    circuit(),
    the_exec_unit(exec_unit),

    // data structures
    dependency_window(issue_buffer_size),
    issue_buffer(issue_buffer_size),
    reg_avail(num_phys_regs, true),
    ready_registers(),
    //    cancel_ranges(num_contexts), // static, init at definition


     //    cancel_ranges(num_contexts), // static, init at definition

    // histogram
    instr_dep_count(4,0),
    histogram_deps(4, histogram_map_t()),
    dispatch_dep_count(4,0),
    histogram_writeback_chain_length(),
    histogram_dependency_window(),
    histogram_issue_buffer(),
    histogram_wakeup_bandwidth(),
    cycle_wakeup_success(0),
    histogram_wakeup_sets(),



    //stats
    wakeup_attempts_total(),
    wakeup_attempts_success(),
    wakeup_attempts_instr_not_ready(),
    wakeup_attempts_wasted(),

    // inputs
    instr_in(),
    rob_head(),
    retiring_instr(),
    branch_exec(),
    mem_access_exec(),
    mem_access_rob(),
    writeback_bus(),
    store_bus(),
    mem_access_stall(),
    syscall_exec(),
    simpanic_in(),

    // outputs
    instr_out(decoder::noop),
    scoreboard_stall(0)
  { 
    // initialize/resize static members after num_contexts is defined
    cancel_ranges = vector<cancel_range_t>(num_contexts);
  }

  // inputs from rename
  inport<decoder::instr_h> instr_in;

  // inputs from decode
  inport<vector<uint64_t> > decode_instr_num;

  // inputs from rob
  inport<vector<uint32_t> > rob_head;        // ptr to instruction ready to retire
  inport<decoder::instr_h>  retiring_instr;  // to clean up cancel_ranges

  // inputs from exec_unit
  inport<branch_packet>     branch_exec;     // branch misprediction info for completing instruction
  inport<mem_access_packet> mem_access_exec; // load/store info for completing instruction
  inport<branch_packet>     branch_rob;
  inport<mem_access_packet> mem_access_rob;  // load/store info for retiring instruction
  inport<bus_packet>        writeback_bus;
  inport<store_bus_packet>  store_bus;

  inport<vector<uint8_t> >  mem_access_stall;  // stall issue of loads/stores due to execution of load/store not recorded in sbc/tsc

  inport<syscall_exec_packet> syscall_exec;  //syscall execution packet
  inport<simpanic_packet>   simpanic_in;
  inport<vector<uint32_t> > frontend_delay_counts; //num of insts from each context in the frontend queue

  // outputs
  statereg<decoder::instr_h> instr_out;
  statereg<uint8_t>          scoreboard_stall;
};

#endif /* SCHEDULER_H */

