#include "rob.h"

rob::rob(arch_model** am) :
  circuit(),
  reorder_buf(rob_size*num_contexts),  //rob_size is a per-context metric 
  count_load_stat(num_contexts, false),
  cancelled_instr_rec(false),
  no_branch(branch_packet()),
  no_mem_access(mem_access_packet()),
  retiring_instr(decoder::noop),

  nspec_miss_cond(0),  
  nspec_miss_return(0),
  nspec_miss_call(0),  
  nspec_branch(0),     
  spec_miss_cond(0),         
  spec_miss_return(0),       
  spec_miss_call(0),  
  spec_branch(0),     

  idle_cycles(0),
    
  the_arch_model(am),
    
  instr_in(),

  branch_exec(),
  store_bus(),
  mem_access_exec(),
  writeback_bus(),
  syscall_exec(), 

  rob_stall(vector<uint8_t>(num_contexts, false)),

  retiring_instr_out(decoder::noop),
  rob_head_timestamp(vector<uint64_t>(num_contexts,0)), // for timestamp cache in exec_unit
  simpanic_out(),  
  instruction_retiring(false),
  retiring_instr_context(0),
  instr_executed(vector<uint64_t>(num_contexts, 0)),
  instr_executed_correctly(vector<uint64_t>(num_contexts, 0))

{

}


/*inline*/ bool rob::stall_prev_stages(uint32_t c) {
  uint32_t num_avail = reorder_buf.num_free_slots_context(c);
  return num_avail < 3;
}

inline bool rob::fifo_empty(uint32_t c) {
  return reorder_buf.is_empty(c);
}

inline void rob::count_executed_instr(uint32_t context, bool canceled) {
  ++(*instr_executed)[context];
  if(!canceled) {
    ++(*instr_executed_correctly)[context];
  }
}

inline void rob::update_branch_exec_stats()
{
  //check for branch_exec and collect stats
  if(branch_exec().is_branch) {
    //Non-speculative branch stats
    if(branch_exec().mispredict) {
      if(branch_exec().is_cond) {
        nspec_miss_cond++;
      } else
        if (branch_exec().is_return) {
          nspec_miss_return++;
        } else
          if (branch_exec().is_call) {
            nspec_miss_call++;
          }
    }
    nspec_branch++;
  }
}


inline void rob::print_debug_info() {
  // per-thread debug information
  if(debug_mask & 0x00000010) {
    printf("ROBRET> rreg %u ret %x freelst %x\n",
           retiring_instr_out()->arch_dest_reg, instruction_retiring(), cancelled_instr_rec);
    for(uint32_t c = 0; c < num_contexts; ++c) {
      uint32_t tail = reorder_buf.tail(c); 
      uint32_t head = reorder_buf.head(c);
      printf("CONTEXT (%x), filled_ROB_slots: %u\n", c, reorder_buf.num_entries(c));
      if(tail != reorder_buf.invalid())
        reorder_buf[tail].instr->print("ROB IN>");
      printf(" Slots: hd %u tl %u stall %x\n", 
             head, tail, (bool)(*rob_stall)[c]);
      if(tail != reorder_buf.invalid()) {
        reorder_buf[head].instr->print("ROBRET>");
        printf(" is cancelled? %d\n", reorder_buf[head].instr->canceled);
      }
      if(head == tail == reorder_buf.invalid())
        printf("\nROB>CONTEXT ROB EMPTY\n");
      printf("CONTEXT (%x) Entire ROB, head to tail:\n", c);
      for (uint32_t temp = head; temp != reorder_buf.invalid(); temp = reorder_buf.get_next(temp) ) {
        reorder_buf[temp].instr->print("");
        printf(" is cancelled? %d\n", reorder_buf[temp].instr->canceled);
      }
    }

    branch_out().print();
    mem_access_out().print();
  }
}

inline void rob::initialize_state() {
  mem_access_out = no_mem_access;  // each cycle, reset mem_access_out to default (no misspeculation, no load/store)
  simpanic_out = simpanic_packet();  //reset to no simpanic
  branch_out = no_branch;
  branch_ret = no_branch;
  retiring_instr_out = decoder::noop;  // eacy cycle, reset retiring_instr to a noop
  retiring_instr = decoder::noop;
  cancelled_instr_rec = false;
  instruction_retiring = false;
}

inline void rob::clear_state_on_simpanic() {
  //clear all contexts' reorder buffers and associated state
  //don't need to send canceled instruction signals to renamer, since renamer 
  //frees up all physical regs after a simpanic
  uint32_t current_context = simpanic_out().context;
  reorder_buf.clear_context(current_context);
}


//function returns true if some kind of mispre/misspec was discovered on a context

inline bool rob::is_context_flush(uint32_t context)
{
  return (simpanic_out().valid  || 
          branch_exec().mispredicted_context(context) ||
          mem_access_exec().mispredicted_context(context) ||
          (syscall_exec().valid && (syscall_exec().context == context) ));
}


//function returns true if there was some kind of mispred/mispec on *ANY* context
inline bool rob::is_context_flush()
{
  return (simpanic_out().valid ||
          branch_exec().mispredict || mem_access_exec().misspeculation ||
          syscall_exec().valid);
}

inline void rob::process_input_instr()
{
  if (!instr_in()->noop) {
    decoder::instr_h the_instr = instr_in();
    
    // if a mispred/misspec completed during the last clock cycle,
    //  then cancel the instruction that just dispatched
    if (is_context_flush(the_instr->context)) {
      the_instr->completed = true;
      the_instr->canceled = true;
      
      //If flush-on-miss, instruction won't be put in ROB,collect stats for this instruction NOW.
      decoder::instr_h inp_instr = instr_in();
    }

    uint32_t c = the_instr->context;
    // 2004.01.19-woley-debugging code: sanity check (pls don't remove)
    uint32_t num_avail = reorder_buf.num_free_slots_context(c);
    assert(num_avail != 0);
    
    //insert instruction into ROB
    rob_entry new_entry(the_instr, no_branch, no_mem_access);
    uint32_t new_slot = reorder_buf.push_tail(c, new_entry);

    reorder_buf[new_slot].instr->reorder_slot = new_slot;
    assert(new_slot!=reorder_buf.invalid());
  }
  else { //if renamer gave a nop this cycle, don't put it into ROB
    //fix stats
  }
}

inline bool rob::schedule_retiring_context(uint32_t *retire_this)
{
  uint32_t context;
  // choose context to retire from, round robin
  context = (retiring_instr_context() + 1) % num_contexts;
  
  bool found_retire_context = false;
  uint32_t end_context = context;
  do {
    if(!fifo_empty(context) && 
       reorder_buf[reorder_buf.head(context)].instr->completed) {
      found_retire_context = true;
      break;
    }
    context = (context + 1) % num_contexts;
  } while (context != end_context);
  
  *retire_this = context;
  return found_retire_context;
}



inline void rob::print_retiring_context_debug_info()
{
#if 0
      {
        uint32_t head = reorder_buf.head(context);
        if(head != reorder_buf.invalid()) {
          decoder::instr_h print_instr = reorder_buf[head].instr;
          printf("%d %08x: %02d %02d %02d\t%llu\t%llu\t%llu\t%llu",
                 context,
                 print_instr->program_counter,
                 print_instr->arch_dest_reg,
                 print_instr->arch_s_reg,
                 print_instr->arch_t_reg,
               print_instr->dispatch_ts,
                 print_instr->issue_ts,
                 print_instr->completion_ts,
                 cycle_count);
          if (print_instr->canceled) printf("\t*");
          printf("\n");
        }
      }
#endif
}


inline void rob::collect_retiring_instr_stats(const rob_entry* the_entry)
{
  // the instruction at the head of the rob is not canceled
  if (the_entry->mem_packet.is_access) {
    if (the_entry->mem_packet.is_store)
      the_retired_store_count++;
    else
      the_retired_load_count++;
    
    if (the_entry->mem_packet.misspeculation)
      the_retired_misspec_store_count++;
  }
}


//function is called when retiring a non-canceled instruction from the head of the ROB of a context
inline void rob::retire_valid_instr(uint32_t context)
{
  uint32_t context_head = reorder_buf.head(context);
  assert(context_head != reorder_buf.invalid());
  rob_entry* retiring_slot = reorder_buf.get_element(context_head);
  collect_retiring_instr_stats(retiring_slot);

  if(debug_mask & 0x00000010) printf("ROBRET> RETIRE CONTEXT (%x)\n", context);
  instruction_retiring = true;
  the_instr_count++;

  //track the last pc to have been retired
  rob_nonspec_final_pc = retiring_slot->instr->program_counter;

  update_cycles_waiting(retiring_slot);

  // compare results with the arch_model
  do_arch_compare(retiring_slot->instr, context);

  if (debug_mask & 0x2000) {
    if (retiring_slot->instr->arch_dest_reg != 0) {
      printf("$%d <- %llx\n", retiring_slot->instr->arch_dest_reg, retiring_slot->instr->wb_data);
    }
  }

  update_branch_predictors_on_instr_retirement(retiring_slot);

  branch_packet bp = retiring_slot->br_packet;
}

inline void rob::update_branch_predictors_on_instr_retirement(const rob_entry* retiring_slot)
{     
  // set up the branch feedback and update stats
  if (retiring_slot->br_packet.is_branch) {
    the_retired_branch_count++;
          
    if (retiring_slot->br_packet.mispredict) {
      the_mispredicted_branch_count++;
    }
  }
}


inline void rob::send_retirement_signals(const rob_entry* retiring_slot)
{
  // instruction at head of rob retired or canceled
  // set up the memory access and branch feedback
  mem_access_out = retiring_slot->mem_packet;
  branch_out     = retiring_slot->br_packet;

  // signal the renamer so that the RRATs and the free list can be updated
  // signal the exec unit so the store buffer can commit its head
  retiring_instr_out = retiring_slot->instr;
  retiring_instr = retiring_slot->instr;
}


inline void rob::process_tornado_warning(bool found_retire_context, uint32_t retiring_context)
{
  if (tornado_warning != 0) {
    // tornado warn if we've executed for several cycles without a real
    // instruction retirement
    if (found_retire_context && !cancelled_instr_rec) {
      idle_cycles = 0;
      if(debug_mask)
        cout<<"Setting idle cycles to 0\n";
    }
    else {
      idle_cycles++;
      if(idle_cycles > tornado_warning) {
        printf("ROB LOCKUP DETECTED AT CYCLE %llu AFTER %u CYCLES WITHOUT A RETIRED INSTR\n", cycle_count, idle_cycles);
        printf("LAST INSTRUCTION RETIRED FROM CONTEXT: %u\n", retiring_instr_context());
        for(uint32_t c=0; c < num_contexts; ++c) {
          printf("CONTEXT (%x) ", c);
          if(!fifo_empty(c))
            reorder_buf[reorder_buf.head(c)].instr->print("ROBHEAD>");
          printf("\n");
        }
        tornado_warning = 0;
        exit(1);
      }
    }
  } // end tornado check

}

inline void rob::update_count_load_stats(const rob_entry* retiring_slot, bool found_retire_context, uint32_t retiring_context)
{
    if (found_retire_context) {
      count_load_stat[retiring_context] = ( (!retiring_slot->instr->canceled)         &&
                                   (!retiring_slot->br_packet.is_branch           ||
                                    !retiring_slot->br_packet.mispredict)         &&
                                   (!retiring_slot->mem_packet.is_access       ||
                                    !retiring_slot->mem_packet.is_store        ||
                                    !retiring_slot->mem_packet.misspeculation) );
    }
}


inline void rob::update_cycles_waiting(const rob_entry* retiring_slot)
{
  //total cycles waiting for retirement post execution
  int64_t cycles_waiting = retiring_slot->instr->completion_ts;
  assert(cycles_waiting != 0);
  cycles_waiting = cycle_count - cycles_waiting;
  assert(cycles_waiting > 0);
  cycles_waiting_to_retire += cycles_waiting;
}

inline void rob::do_arch_compare(const decoder::instr_h instr_retiring, uint32_t retiring_context)
{
  bool arch_problem = false;
  
  decoder::instr_h arch_instr;
  do {
    // retire an instruction from the arch model that corresponds to the thread
    // that owns the instruction that is retiring from the rob
    arch_instr = the_arch_model[retiring_context]->do_cycle();
    
    //fprintf(stderr, "ROB retiring PC %x\n", arch_instr->program_counter);
    
  } while(!arch_instr->instr); // discard arch-model's retiring no-ops
  
  if (debug_mask & 0x00002000) {
    printf("CONTEXT (%x) ", retiring_context);
    printf("retiring %08x\t%08x\t%x", arch_instr->program_counter, arch_instr->instr, arch_instr->opcode);
    arch_instr->print("\nARCH_RETIRE>");
    reorder_buf[reorder_buf.head(retiring_context)].instr->print("\nPIPE_RETIRE>");
    printf("\n");
    reorder_buf[reorder_buf.head(retiring_context)].br_packet.print();
    pretty_print(arch_instr);
    printf("\n");
    fflush(stdout);
  }
  
  // verify that arch model and pipelined model retired the same instruction
  if(arch_instr->program_counter != instr_retiring->program_counter) {
    printf("ARCH_COMPARE> MISMATCHED INSTRUCTION PC\n");
    arch_problem = true;
  }
  if((arch_instr->arch_dest_reg != 0) || instr_retiring->is_store) {
    if(arch_instr->is_store || arch_instr->is_load) {
      if(arch_instr->vaddr != instr_retiring->vaddr) {
        printf("ARCH_COMPARE> INCORRECT MEMORY ACCESS ADDRESS ");
        printf("ARCH vaddr %x VS. ROB vaddr %x\n", arch_instr->vaddr, 
               instr_retiring->vaddr);
        arch_problem = true;
      }
    }
            
    if(arch_instr->wb_data != instr_retiring->wb_data) {
      printf("ARCH_COMPARE> INCORRECT WB_DATA: ");
      printf("$%d <- %llx", arch_instr->arch_dest_reg, arch_instr->wb_data);
      printf(" VS. ROB: $%d <- %llx\n", instr_retiring->arch_dest_reg, instr_retiring->wb_data);
      arch_problem = true;
    }
  }

  // if arch model and pipelined model do not agree, exit
  if(arch_problem) {
    printf("CONTEXT (%x) ", retiring_context);
    arch_instr->print("ARCH_INSTR>");
    printf("\n");
    printf("CONTEXT (%x) ", retiring_context);
    instr_retiring->print("ROBRET_INSTR>");
    printf("\n");
    printf("Cycle: %llu\n", cycle_count);
    if(skip_retire_check_failures) { //ignore retire_check_failure, execute a resync
      vector<uint64_t> arch_reg_file;
      arch_reg_file =   the_arch_model[retiring_context]->get_reg_file();
      simpanic_out = simpanic_packet(instr_retiring, the_arch_model[retiring_context]->get_pc(), arch_reg_file);
      num_retire_check_failures++;
      printf("\nArch_Reg_File on arch_compare mismatch = \n");
      for(uint32_t reg=0; reg<NUM_ARCH_REGS; reg++) {
        printf("Reg %u Val %llu   ", reg, arch_reg_file[reg]);
        if(reg%6==0)
          printf("\n");
      }
      printf("\n");

      //set instr_retiring to noop so rename doesn't try to free a register based on this
      //note: retiring_instr is a statereg
      retiring_instr_out = decoder::noop;
      retiring_instr = decoder::noop;
    }
    else {
      exit(1);
    }
  }
  //retiring_instr_out = retiring_instr;
}


inline void rob::retire_canceled_instr(uint32_t context)
{
  uint32_t context_head = reorder_buf.head(context);
  assert(context_head != reorder_buf.invalid());
  // instr at head of rob completed and canceled
  if (reorder_buf[context_head].mem_packet.is_access &&
      !reorder_buf[context_head].mem_packet.is_store &&
      reorder_buf[context_head].mem_packet.misspeculation &&
      count_load_stat[context]) {
    the_retired_misspec_load_count++;
  }
  
  if(debug_mask & 0x00000010) printf("ROBRET> RETIRE OF CANCELED INSTR CONTEXT (%x)\n", context);
        
}



//this function finds the recovery context and recovery instr_num in case of a misspec
inline bool rob::investigate_misspec(uint32_t* misspec_context, uint64_t* cancel_instr_num)
{
  bool misspeculation = false;

  if(simpanic_out().valid) {
    misspeculation = true;
    *misspec_context = simpanic_out().context;
    *cancel_instr_num = simpanic_out().instr_num + 1;
  }
  else if (mem_access_exec().misspeculation) {
    misspeculation = true;
    *misspec_context = mem_access_exec().recovery_context;
    *cancel_instr_num = mem_access_exec().recovery_ts;
  }
  else if(branch_exec().mispredict) {
    misspeculation = true;
    *misspec_context = branch_exec().context;
    *cancel_instr_num = branch_exec().instr_num + 1;
  }
  else if(syscall_exec().valid) {
    misspeculation = true;
    *misspec_context = syscall_exec().context;
    *cancel_instr_num = syscall_exec().instr_num + 1;
  }

  if(debug_mask & 0x10) printf("ROB>: Saw misspeculation on context %u, instrnum %llu\n", *misspec_context, *cancel_instr_num);

  return misspeculation;
}


//flush a context, because of context kill, or misspeculation on context
//context_kill = true when context was KILLED. if misspec/mispred, context_kill = false
//cancel_ts = all instructions with this AND larger TimeStamps should be canceled. 
inline void rob::flush_context(uint32_t context, uint64_t cancel_ts)
{
  uint32_t slot = reorder_buf.tail(context);  //traversing from tail to head. 
  uint32_t prev_slot;
  bool finished = false;
  while (slot != reorder_buf.invalid() && !finished) {
    prev_slot = reorder_buf.get_prev(slot);

    if(reorder_buf[slot].instr->instr_num >= cancel_ts) {
      decoder::instr_h the_instr = reorder_buf[slot].instr;


      // remove wrong path instructions from the correct exec count
      if(!the_instr->canceled && the_instr->completed) {
        --(*instr_executed_correctly)[context];
      }

      // free the instruction's physical register, and update nop stats

      // cancel the instruction
      the_instr->completed = true;
      the_instr->canceled = true;

      reorder_buf[slot].br_packet.cancelled = true;
      reorder_buf[slot].mem_packet.cancelled = true;
    }
    else {
      finished = true;  //when the first non-flushed instruction is found, you're done.
    }
    slot = prev_slot;
    if(debug_mask & 0x10) printf("ROB_FLUSH> Next slot = %u\n", slot);
  }
}


//mark instructions as completed if they're on the WB bus
inline void rob::process_wb_bus()
{
  if (writeback_bus().reorder_slot != reorder_buf.invalid()) {
    uint32_t c = writeback_bus().context;
    decoder::instr_h the_instr = reorder_buf[writeback_bus().reorder_slot].instr;
    count_executed_instr(c, the_instr->canceled);
    the_instr->completed = true;
    //if instruction was completed when context was non-speculative (stat collection)

    uint32_t wb_phys_reg = writeback_bus().reg_tag;
    if (wb_phys_reg != the_instr->phys_dest_reg) {
      printf("ASSERT: instr ts %llu sb_slot %u reorder_slot %u writeback_bus_reg %u rob_phys_reg %u \n", 
             the_instr->instr_num, writeback_bus().sb_slot, writeback_bus().reorder_slot,
             wb_phys_reg, the_instr->phys_dest_reg); 
    }
    assert(wb_phys_reg == the_instr->phys_dest_reg);
    the_instr->wb_data = writeback_bus().data;
    the_instr->completion_ts = cycle_count;
  }
}


//mark completed branches
inline void rob::process_branch_exec()
{
  if (branch_exec().is_branch) {
    assert(branch_exec().reorder_slot != reorder_buf.invalid());
    uint32_t c = branch_exec().context;
    decoder::instr_h the_instr = reorder_buf[branch_exec().reorder_slot].instr;
    
    the_instr->mispredict = branch_exec().mispredict;
    
    // complete the instruction only if it is *not* a jal or jalr:
      // jals and jalrs complete when they get a signal from the writeback bus.
    if (the_instr->arch_dest_reg == 0) {
      count_executed_instr(c, the_instr->canceled);
      the_instr->completed = true;
      the_instr->completion_ts = cycle_count;
    }
    
    if (alpha_renaming || !the_instr->canceled) {
      reorder_buf[branch_exec().reorder_slot].br_packet = branch_exec();
    }
  }
}


inline void rob::process_store_bus()
{
  if (store_bus().reorder_slot != reorder_buf.invalid()) {
    uint32_t c = store_bus().context;
    decoder::instr_h the_instr = reorder_buf[store_bus().reorder_slot].instr;
    count_executed_instr(c, the_instr->canceled);
    the_instr->completed = true;
    the_instr->completion_ts = cycle_count;
  }

}

inline void rob::process_syscall_exec()
{
  if (syscall_exec().valid) {
    assert(syscall_exec().reorder_slot != reorder_buf.invalid());
    uint32_t c = syscall_exec().context;
    decoder::instr_h the_instr = reorder_buf[syscall_exec().reorder_slot].instr;
    count_executed_instr(c, the_instr->canceled);
    the_instr->completed = true;
    the_instr->completion_ts = cycle_count;
  }
}


inline void rob::process_mem_exec()
{
  if (mem_access_exec().is_access) {
    decoder::instr_h the_instr = reorder_buf[mem_access_exec().rob_slot].instr;
    if (alpha_renaming || !the_instr->canceled) {
      reorder_buf[mem_access_exec().rob_slot].mem_packet = mem_access_exec();
      reorder_buf[branch_exec().reorder_slot].br_packet = branch_exec();
    }
  }
}


inline void rob::generate_rob_head_ts()
{
  for(uint32_t c=0; c < num_contexts; ++c) {
    (*rob_stall)[c] = stall_prev_stages(c);
    uint32_t head_of_rob = reorder_buf.head(c);
    if(head_of_rob != reorder_buf.invalid())
      (*rob_head_timestamp)[c] = reorder_buf[head_of_rob].instr->instr_num; // for timestamp cache
    else
      (*rob_head_timestamp)[c] = 0;
  }
}
 
inline void rob::generate_rob_head_and_tail()
{
  vector<uint32_t> temp_head(num_contexts);
  vector<uint32_t> temp_tail(num_contexts);
  for(uint32_t c=0; c<num_contexts; c++) {
    temp_head[c] = reorder_buf.head(c);
    temp_tail[c] = reorder_buf.tail(c);
  }
  rob_head = temp_head;
  rob_tail = temp_tail;
}


inline void rob::recalc() {
  print_debug_info();
  initialize_state();
  //if simpanic, clear all queues, neglect instruction at input latch.
  if(simpanic_out().valid) {
    clear_state_on_simpanic();
    return;
  }

  //state: no simpanic
  process_input_instr();  // insert instructions into tail of queue as they are dispatched from renamer:
  update_branch_exec_stats();  //collect stats from executed branch

  //context_flush: some context had a mispred/misspec OR got killed in the last cycle
  if (!is_context_flush()) {
    //No flush. retire completed instruction from head of queue.  
    uint32_t retiring_context;               // Retirement context
    bool found_retire_context = schedule_retiring_context(&retiring_context);    //choose a context to retire from.
    rob_entry* retiring_slot;
    
    if(debug_mask & 0x000000010) printf("ROB> State: no context flush\n");

    // retire completed instructions from the head of queue:
    if (found_retire_context) { //if there IS an instruction to retire
      print_retiring_context_debug_info();
      uint32_t context_head = reorder_buf.head(retiring_context);
      assert(context_head!=reorder_buf.invalid());
      retiring_slot = &reorder_buf[context_head];
      if (!reorder_buf[context_head].instr->canceled) {
        retire_valid_instr(retiring_context);
      }
      else {
        retire_canceled_instr(retiring_context);
        cancelled_instr_rec = true;
      }
      send_retirement_signals(retiring_slot);
    } // end if [instr available to retire]
    else {
      if(debug_mask & 0x000000010) printf("ROBRET> NO INSTRUCTION RETIRED\n");
    }

    process_tornado_warning(found_retire_context, retiring_context); //check for ROB stall
    if(found_retire_context) {
      update_count_load_stats(retiring_slot, found_retire_context, retiring_context);
      //do the actual retirement
      reorder_buf.pop_head(retiring_context, &retiring_slot);    
      retiring_instr_context = retiring_context;
    }
  } // end if [last instruction completed by exec unit was not a mispredicted branch or misspeculated load/store]
  else {
    // a context flush has happened. We won't retire an instruction(don't know why??) We'll just focus on handling the flush.
    //      idle_cycles = 0;
    if(debug_mask & 0x000000010) printf("ROB> State: context flush\n");
    uint64_t cancel_instr_num=0;
    uint32_t misspec_context=0;
    bool misspeculation = investigate_misspec(&misspec_context, &cancel_instr_num);

    for(uint32_t c = 0; c < num_contexts; ++c) {
      if((misspeculation && (c == misspec_context))) {      
        uint64_t cancel_ts;  //what instructions should I flush
        cancel_ts = cancel_instr_num;

        if(debug_mask & 0x00000010) printf("ROB KILL> context %u misspec: %x ts: %llu\n",
                                           c, (misspeculation && (c == misspec_context)), cancel_ts);
        flush_context(c, cancel_ts);
      }
    }  
  } // end else [context_flush] happened
  
  // mark any instructions that were completed by execution unit (including loads; excluding branches and stores)
  process_wb_bus();
  process_branch_exec();
  process_store_bus();
  process_syscall_exec();
  process_mem_exec();

  generate_rob_head_ts();  //generate stuff needed for ts-cache
  generate_rob_head_and_tail();  //generate the statereg with updated rob head and tail pointers
}

// used as part of a system-wide check to ensure that no physical registers have been lost
void rob::account_phys_regs(vector<uint32_t>& phys_regs) {
  for(uint32_t c = 0; c < num_contexts; ++c) {
    for (uint32_t rob_slot = reorder_buf.head(c); rob_slot != reorder_buf.invalid(); rob_slot = reorder_buf.get_next(rob_slot))
      phys_regs[reorder_buf[rob_slot].instr->phys_dest_reg] = 1;
  }
  phys_regs[instr_in()->phys_dest_reg] = 1;
  //add retiring_instr_out()->phys_dest_reg to the list
  if(retiring_instr->completed)
    phys_regs[retiring_instr->phys_dest_reg] = 1;
}

