#ifndef RENAME_STAGE_H
#define RENAME_STAGE_H

#include "circuit.h"
#include "regfile_circuit.h"
#include "packets.h"
#include "rename_tables.h"
#include "free_list.h"
#include <algorithm>
#include <vector>
#include <deque>
#include <iostream>
#include "globals.h"
#include <map>
using namespace std;

// SMT
// 1. One RAT and one RRAT per thread.
//    When renaming an instr, select RAT based on arriving instr's context.
//    When retiring an instr, select RRAT based on retiring instr's context.
//    When handling a retired mispredict, select RAT and RRAT based on mispredicted branch's context.

// Alpha Style Renaming
// 1. For a load/store, create the checkpoint BEFORE renaming the instruction, because
//    that instruction will be renamed again in the event of a misspeculation.
// 2. For a branch, create the checkpoint AFTER renaming the instruction, because
//    that instruction will not be renamed again in the event of a misprediction.
// 3. Each branch/load/store instruction carries a rat_checkpt index (the index
//    of the rat checkpoint for that instruction). When a misprediction/misspeculation
//    is detected, the renamer resets the active rat to the designated checkpoint.

extern uint32_t num_phys_regs;    // commandline parameter, default is 64
extern uint32_t num_contexts;       // commandline parameter, default is 1
extern uint32_t rob_size;         // commandline parameter, default is 64
extern uint32_t debug_mask;       // rename stage debug_mask is 0x00000004
                                  // RAT and RRAT debug_mask is 0x00000080
                                  // free list debug_mask is    0x00040000
                                  //reference counts: 0x40000
extern bool bullet_proof_check_cycle;
extern uint64_t cycle_count;
extern bool alpha_renaming;

class rename_stage : circuit
{
 private:

  uint32_t index(uint32_t context, uint32_t rat_index) const;
  void free_register(uint32_t phys_reg);
  void get_register(uint32_t& phys_reg);
  void free_registers_used(uint32_t context);
  void mark_registers_used(uint32_t context);
  void print_debug_info();  //print stats when debug flag(s) are set
  void bullet_proof_check(); //check for physical register allocation
  void initialize_rats(uint32_t starting_context); //at the beginning of execution, initializes all the RATs and the free list
  void handle_simpanic(simpanic_packet the_simpanic_packet); 
  void handle_stall(); //if there's a stall, output nop

  void recalc();
  bool can_rename_instr();         //should i rename an instruction?
  void flush_input_instr();        //in case of mispredict
  void rename_input_instr(); 
  void process_retiring_instr();   //handle instruction retiring from the ROB
  void process_misspeculation();   //repair RAT state on misspeculation
  void stall_if_needed();          //stall machine if out of resources (physical regs, mostly)

 public:
  void account_phys_regs(vector<uint32_t>& phys_regs) const;
  
  // inits
  rename_stage(const vector<uint64_t> *rf);

  ~rename_stage() {
    delete [] active_rat;
  }
  
  // inputs    
  inport<decoder::instr_h>  instr_in;          // input from store set stage
  inport<branch_packet>     branch_exec;       // branch misprediction info for completing instruction
  inport<mem_access_packet> mem_access_exec;   // load/store info for completing instruction
  inport<branch_packet>     branch_rob;
  inport<mem_access_packet> mem_access_rob;
  inport<uint8_t>           scoreboard_stall;
  inport<vector<uint8_t> >  rob_stall;
  inport<decoder::instr_h>  retiring_instr;    // free list info for retiring instructions
  inport<syscall_exec_packet> syscall_exec;  //syscall execution packet from exec unit
  inport<simpanic_packet> simpanic_in;  //recovery from retire-check failures
  
  // outputs
  statereg<decoder::instr_h> instr_out;
  statereg<uint8_t>             rename_stall;

  vector<uint32_t> reg_kill_count;
  
  // renamer tables
  vector<rename_state>      the_rat;
  vector<rename_state>      the_rrat;
  free_list        FL;
  
  //    vector<uint32_t > active_rat;
  uint32_t *active_rat;
  vector<uint32_t> reg_access_count;
  
  bool     free_reg_stall;
  bool     p4rename_stall;
  uint64_t earliest_recovery_ts;

  //simpanic recovery state
  bool simpanic_recovery_mode; //true when you saw a simpanic and were waiting for the rob to become empty
};


#endif /* RENAME_STAGE_H */
