#include "stdio.h"
#include "pretty_print.h"
#include "decode.h"

static
char* opcode_names[] = {
  "sll",
  "srl",
  "sra",
  "sllv",
  "srlv",
  "srav",
  "dsll",
  "dsrl",
  "dsra",
  "dsllv",
  "dsrlv",
  "dsrav",
  "jr",
  "jalr",
  "mull",
  "div",
  "divu",
  "mulh",
  "mulhu",
  "dmull",
  "ddiv",
  "ddivu",
  "dmulh",
  "dmulhu",
  "addu",
  "subu",
  "uand",
  "uor",
  "uxor",
  "nor",
  "daddu",
  "dsubu",
  "slt",
  "sltu",
  "beq",
  "bne",
  "addiu",
  "slti",
  "sltiu",
  "andi",
  "ori",
  "xori",
  "aui",
  "daddiu",
  "lb",
  "lh",
  "lw",
  "ld",
  "lbu",
  "lhu",
  "lwu",
  "sb",
  "sh",
  "sw",
  "sd",
  "bltz",
  "blez",
  "bgez",
  "bgtz",
  "j",
  "jal",
  "add_s",
  "sub_s",
  "mul_s",
  "div_s",
  "abs_s",
  "neg_s",
  "trunc_l_s",
  "cvt_s_l",
  "cvt_l_s",
  "trunc_w_s",
  "c_eq_s",
  "c_lt_s",
  "c_le_s",
  "add_d",
  "sub_d",
  "mul_d",
  "div_d",
  "abs_d",
  "neg_d",
  "trunc_l_d",
  "cvt_d_l",
  "cvt_l_d",
  "trunc_w_d",
  "c_eq_d",
  "c_lt_d",
  "c_le_d",
  "cvt_s_d",
  "cvt_d_s",
  "syscall_halt",
  "syscall_unix",
  "syscall_ftostr",
  "syscall_strtof",
  "signaling_noop"
};

static
void
print_shift_imm(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d %d $%02d",
         instr->arch_dest_reg,
         instr->shamt_immediate(),
         instr->arch_t_reg);
}

static
void
print_rrr(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d $%02d $%02d",
         instr->arch_dest_reg,
         instr->arch_s_reg,
         instr->arch_t_reg);
}

static
void
print_jr(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d", instr->arch_s_reg);
}

static
void
print_branch(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d $%02d %d",
         instr->arch_s_reg,
         instr->arch_t_reg,
         instr->branch_immediate());
}

static
void
print_simm(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d $%02d 0x%08x",
         instr->arch_dest_reg,
         instr->arch_s_reg,
         instr->signed_immediate32());
}

static
void
print_uimm(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d $%02d 0x%08x",
         instr->arch_dest_reg,
         instr->arch_s_reg,
         instr->unsigned_immediate());
}

static
void
print_st(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "$%02d $%02d %d",
         instr->arch_s_reg,
         instr->arch_t_reg,
         instr->signed_immediate32());
}

static
void
print_j(decoder::instr_h instr, FILE *outfile) {
  fprintf(outfile, "0x%08x",
         instr->jump_immediate());
}

void
pretty_print(decoder::instr_h instr, FILE* outfile)
{
//  fprintf(outfile, "%s\t0x%x\t", opcode_names[instr->opcode], instr->opcode);
  fprintf(outfile, "%s\t", opcode_names[instr->opcode]);
  switch (instr->opcode)
  {
  case decoder::sll:
  case decoder::srl:
  case decoder::sra:
  case decoder::dsll:
  case decoder::dsrl:
  case decoder::dsra:
    print_shift_imm(instr, outfile);
    break;
  case decoder::sllv:
  case decoder::srlv:
  case decoder::srav:
  case decoder::dsllv:
  case decoder::dsrlv:
  case decoder::dsrav:
    print_rrr(instr, outfile);
    break;
  case decoder::jr:
  case decoder::jalr:
    print_jr(instr, outfile);
    break;
  case decoder::mull:
  case decoder::div:
  case decoder::divu:
  case decoder::mulh:
  case decoder::mulhu:
  case decoder::dmull:
  case decoder::ddiv:
  case decoder::ddivu:
  case decoder::dmulh:
  case decoder::dmulhu:
  case decoder::addu:
  case decoder::subu:
  case decoder::uand:
  case decoder::uor:
  case decoder::uxor:
  case decoder::nor:
  case decoder::daddu:
  case decoder::dsubu:
  case decoder::slt:
  case decoder::sltu:
    print_rrr(instr, outfile);
    break;
  case decoder::beq:
  case decoder::bne:
    print_branch(instr, outfile);
    break;
  case decoder::addiu:
  case decoder::daddiu:
  case decoder::slti:
  case decoder::sltiu:
    print_simm(instr, outfile);
    break;
  case decoder::andi:
  case decoder::ori:
  case decoder::xori:
  case decoder::aui:
    print_uimm(instr, outfile);
    break;
  case decoder::lb:
  case decoder::lh:
  case decoder::lw:
  case decoder::ld:
  case decoder::lbu:
  case decoder::lhu:
  case decoder::lwu:
    print_simm(instr, outfile);
    break;
  case decoder::sb:
  case decoder::sh:
  case decoder::sw:
  case decoder::sd:
    print_st(instr, outfile);
    break;
  case decoder::bltz:
  case decoder::blez:
  case decoder::bgez:
  case decoder::bgtz:
    print_branch(instr, outfile);
    break;
  case decoder::j:
  case decoder::jal:
    print_j(instr, outfile);
    break;
  case decoder::add_s:
  case decoder::sub_s:
  case decoder::mul_s:
  case decoder::div_s:
  case decoder::abs_s:
  case decoder::neg_s:
  case decoder::trunc_l_s:
  case decoder::cvt_s_l:
  case decoder::cvt_l_s:
  case decoder::c_eq_s:
  case decoder::c_lt_s:
  case decoder::c_le_s:
  case decoder::add_d:
  case decoder::sub_d:
  case decoder::mul_d:
  case decoder::div_d:
  case decoder::abs_d:
  case decoder::neg_d:
  case decoder::trunc_l_d:
  case decoder::cvt_d_l:
  case decoder::cvt_l_d:
  case decoder::c_eq_d:
  case decoder::c_lt_d:
  case decoder::c_le_d:
  case decoder::cvt_s_d:
  case decoder::cvt_d_s:
    print_rrr(instr, outfile);
    break;
  case decoder::syscall_halt:
    break;
  case decoder::syscall_unix:
    print_rrr(instr, outfile);
    break;
  case decoder::syscall_ftostr:
    print_rrr(instr, outfile);
    break;
  case decoder::syscall_strtof:
    print_rrr(instr, outfile);
    break;
  case decoder::signaling_noop:
//    fprintf(outfile, "error: *unknown opcode*!!!!\n");
    break;
  default:
//    fprintf(outfile, "error: unknown opcode!!!!!\n");
    break;
  }
}
