#ifndef _PACKETS_H_GUARD
#define _PACKETS_H_GUARD

#include <bitset>
#include <iostream>
#include "globals.h"
#include "decode.h" 


extern uint32_t num_contexts;

struct store_bus_packet {
  size_t   sb_slot;
  uint32_t reorder_slot;
  uint32_t context;
  uint64_t instr_num;

  store_bus_packet(size_t sbs = 0, uint32_t rs = INV_POOL_PTR, uint32_t c = 0, uint64_t inum = 0) :
    sb_slot(sbs), reorder_slot(rs), context(c), instr_num(inum)
    { }

  bool operator==(const store_bus_packet& b) {
    return ((sb_slot      == b.sb_slot) &&
            (reorder_slot == b.reorder_slot) &&
            (context      == b.context) &&
            (instr_num    == b.instr_num));
  }
};

struct bus_packet {
  uint32_t  reg_tag;
  uint64_t data;
  size_t   sb_slot;
  uint32_t reorder_slot;
  uint32_t context;
  uint64_t instr_num;

  bus_packet(uint32_t t = 0,  uint64_t d = 0, size_t sbs = 0,
             uint32_t rs = INV_POOL_PTR, uint32_t c = 0, uint64_t inum = 0) :
    reg_tag(t), data(d), sb_slot(sbs), reorder_slot(rs), context(c), instr_num(inum)
    { }

  bool operator==(const bus_packet& b) {
    return ((reg_tag      == b.reg_tag) &&
            (data         == b.data) &&
            (sb_slot      == b.sb_slot) &&
            (reorder_slot == b.reorder_slot) &&
            (context      == b.context) &&
            (instr_num    == b.instr_num));
  }
};

//shuts stl up
inline bool operator<(const bus_packet& __a, const bus_packet& __b) {
  return (__a.instr_num < __b.instr_num);
}


struct branch_ret_packet {
  bool valid;
  uint32_t pc;
  vector<uint8_t> history;
  bool taken;
};

struct branch_packet {
  bool     is_branch;
  bool     is_cond;
  bool     taken;
  bool     mispredict;
  bool     is_return;
  bool     is_call;
  bool     local_prediction;
  bool     global_prediction;
  uint32_t correct_target;
  uint32_t instr_pc;

  long_vector_c global_history;
  long_vector_c   path_history;

  uint32_t ra_stack_ptr;
  uint32_t ra_stack_head;
  uint32_t predicted_pc;
  size_t   sb_slot;
  uint32_t reorder_slot;
  uint32_t context;
  uint32_t rat_checkpt;  //checkpoint to restore the rat to
  uint64_t instr_num;
  bool     cancelled;

  branch_packet() : 
    is_branch(false), 
    is_cond(false),
    taken(false), 
    mispredict(false), 
    is_return(false), 
    is_call(false),
    local_prediction(false), 
    global_prediction(false), 
    correct_target(0), 
    instr_pc(0),
    global_history(), 
    path_history(), 
    ra_stack_ptr(0), 
    ra_stack_head(0), 
    predicted_pc(0),
    sb_slot(0), 
    reorder_slot(INV_POOL_PTR), 
    context(0), 
    rat_checkpt(0), 
    instr_num(0),
    cancelled(false)
  { }
  
  inline bool mispredicted_branch() const { return is_branch && mispredict; }
  // for smt/polyflow instr that are not branches can be mispredicted as branches
  inline bool mispredicted_context(uint32_t c) const { return /*is_branch && */mispredict && (c == context); }

  void print() const {
    //FIXME: print ghist
    printf("branch packet: ib %d iret %d ical %d tak %d mis %d lpre %d gpre %d ctarg %x ptarg %x pc %x slot %d ghist -- ratchk %u  ts %llu ctext %d canc %u\n",
           is_branch, is_return, is_call, taken, mispredict, local_prediction, global_prediction,
           correct_target, predicted_pc, instr_pc, reorder_slot,  rat_checkpt, instr_num, context, cancelled);
  }
};

struct store_packet {
  bool   is_store;
  size_t sb_slot;
  uint32_t reorder_slot;
  uint64_t data;
  uint32_t context;

  store_packet() :
    is_store(false),
    sb_slot(0),
    reorder_slot(INV_POOL_PTR),
    data(0),
    context(0)
    { }

  void print() const {
    printf("store packet: is %d, sb_slot %d, rob_slot %d, ctext %d\n", is_store, sb_slot, reorder_slot, context);
  }
};


struct mem_access_packet {
  bool     is_access;
  bool     is_store;      // 1 if store, 0 if load
  bool     cache_access;
  uint32_t mem_address;
  uint32_t instr_pc;
  uint64_t instr_num;
  bool     misspeculation;
  bool     replay;
  uint32_t context;
  size_t   sb_slot;
  size_t   rob_slot;
  uint32_t rat_checkpt;  //checkpoint to restore the rat to
  uint32_t size;
  bool     in_sbc;
  bool     in_tsc;
  bool     cancelled;
  bool     l1_hit;
  bool     l2_hit;
  uint32_t recovery_pc;
  uint64_t recovery_ts;
  uint32_t recovery_context;
  uint32_t prev_pc;
  bool     lsq_search_penalty;

  mem_access_packet() : 
    is_access(false),
    is_store(false),
    cache_access(false),
    mem_address(0xffffffff),
    instr_pc(0xffffffff),
    instr_num(0xffffffffffffffffULL),
    misspeculation(false),
    replay(false),
    context(~0),
    sb_slot((size_t)-1),
    rob_slot((size_t)-1),
    rat_checkpt(0),
    size(0),
    in_sbc(false),
    in_tsc(false),
    cancelled(false),
    l1_hit(true),
    l2_hit(true),
    recovery_pc(0xffffffff), // pc where fetch should restart
    recovery_ts(0xffffffffffffffffULL), // ts where flushing should begin
    recovery_context(~0),
    prev_pc(~0),
    lsq_search_penalty(false)
  { }

  bool mispredicted_context(uint32_t c) const { return misspeculation && (c == recovery_context); }
  
  void print() const {
    printf("mem_access packet: isa %d iss %d cache %d addr %08x pc %08x instr_num %llu mis %d rep %d ctext %d sb_slot %u rob_slot %u "
           "ratchk %u size %u canc %u rec_pc %08x rec_ts %llu \n", is_access, is_store, cache_access,
           mem_address, instr_pc, instr_num, misspeculation, replay, recovery_context, sb_slot, rob_slot, rat_checkpt, size, cancelled,
           recovery_pc, recovery_ts);
  }
};

class syscall_exec_packet { 
 public:
  bool valid;
  uint32_t program_counter;
  uint32_t context;
  bool cancelled;
  size_t sb_slot;
  uint32_t reorder_slot;
  uint32_t rat_checkpt; 
  uint64_t instr_num;
  uint32_t ra_stack_ptr;
  uint32_t ra_stack_head;
  long_vector_c global_history;

 public:
  syscall_exec_packet(): 
    valid(false), 
    program_counter(0xffffffff), 
    context(0xffffffff),
    cancelled(false), 
    sb_slot(~0), 
    reorder_slot(INV_POOL_PTR), 
    rat_checkpt(~0), 
    instr_num(~0),
    ra_stack_ptr(~0), 
    ra_stack_head(~0), 
    global_history(400)
  { }

  syscall_exec_packet(decoder::instr_h the_instr): 
    valid(true), 
    program_counter(the_instr->program_counter), 
    context(the_instr->context), 
    cancelled(the_instr->canceled), 
    sb_slot(the_instr->sb_slot), 
    reorder_slot(the_instr->reorder_slot), 
    rat_checkpt(the_instr->rat_checkpt), 
    instr_num(the_instr->instr_num),
    ra_stack_ptr(the_instr->ra_stack_ptr), 
    ra_stack_head(the_instr->ra_stack_head), 
    global_history(the_instr->global_history)
  { }

  void print() const { printf("Syscall packet: valid %u   pc %08x   instr_num %llu  context %d \n", valid, program_counter, instr_num, context);  }
};


//the simpanic packet. 
class simpanic_packet {
 public:
  bool valid;
  uint32_t program_counter;  //pc of the mismatched retired instruction
  uint32_t next_pc; //program counter to start fetching from
  uint32_t context;
  size_t sb_slot;
  uint32_t reorder_slot;
  uint64_t instr_num; //instruction number of mismatched instruction
  uint32_t ra_stack_ptr;
  uint32_t ra_stack_head;
  uint32_t rat_checkpt;
  long_vector_c global_history;

  //a simpanic is usually accompanied with an incorrect value written to a particular
  vector<uint64_t> reg_file;

 public:
  simpanic_packet(): 
    valid(false), 
    program_counter(0xffffffff), 
    next_pc(0),
    context(0xffffffff),
    sb_slot(~0), 
    reorder_slot(INV_POOL_PTR), 
    instr_num(~0),
    ra_stack_ptr(~0), 
    ra_stack_head(~0), 
    rat_checkpt(0),
    global_history(400), 
    reg_file()
  { }

  simpanic_packet(decoder::instr_h the_instr, uint32_t x_next_pc, 
                  vector<uint64_t> x_reg_file): 
    valid(true), 
    program_counter(the_instr->program_counter), 
    next_pc(x_next_pc),
    context(the_instr->context), 
    sb_slot(the_instr->sb_slot), 
    reorder_slot(the_instr->reorder_slot), 
    instr_num(the_instr->instr_num), 
    ra_stack_ptr(the_instr->ra_stack_ptr), 
    ra_stack_head(the_instr->ra_stack_head), 
    rat_checkpt(the_instr->rat_checkpt),
    global_history(the_instr->global_history), 
    reg_file(x_reg_file)
  { }

    void print() const { printf("Simpanic packet: valid %u   pc %08x   next_pc %08x  instr_num %llu   context %d  \n", valid, program_counter, next_pc, instr_num, context);  }
    inline bool mispredicted_context(uint32_t c) const {return valid && (c==context);}
};



#endif
