#ifndef GSHARE_H
#define GSHARE_H

#include <inttypes.h>
#include "long_vector.h"

using namespace std;                                                                                                              
class gshare
{
  uint32_t* table;
  const uint32_t addr_bits;
  const uint32_t history_bits;
  const uint32_t table_size;

  uint32_t addr_word(uint32_t hval) {
    return (hval >> 4);
  }

  uint32_t addr_bit(uint32_t hval) {
    return (hval << 1) & 0x1f;
  }

  uint32_t get_table_entry(uint32_t hval) {
    return (table[addr_word(hval)] >> addr_bit(hval)) & 0x3;
  }

  uint32_t get_hval(const uint32_t& addr, const long_vector_c& gh) {
    int ghist = 0;

    /*
    for (uint32_t i=0; (i < history_bits)&& (i <global_history.size()); i++ )
      ghist |= (global_history[ global_history.size() -1 -i] & 0x1 ) << i;
    */

    for (int32_t i = history_bits-1; i>=0; i--) {
      ghist <<= 1;
      ghist  |= gh[i];
    }

    const uint32_t GOLDEN_RATIO = 0x9e3779b9;
    uint32_t addrhash = ((addr >> 2) * GOLDEN_RATIO) >> (32 - addr_bits);
    return (addrhash ^ (ghist & ((1 << history_bits) - 1))) & (table_size - 1);
  }

  void set_table_entry(uint32_t hval, uint32_t new_val) {
    uint32_t old_val = table[addr_word(hval)];
    uint32_t old_mask = 0x3 << addr_bit(hval);
    uint32_t new_mask = (new_val & 0x3) << addr_bit(hval);
    table[addr_word(hval)] = ((old_val & ~old_mask) | new_mask);
  }

  uint32_t next_state(uint32_t state, uint32_t taken) {
    // transition table:
    // oldstate input newstate
    // 00         0      00
    // 00         1      01
    // 01         0      00
    // 01         1      11
    // 10         0      00
    // 10         1      11
    // 11         0      10
    // 11         1      11
    // so the lookuptable holds the bits:
    // 1110 1100 1100 0100
    const uint32_t LUT = 0x0000ecc4;
    uint32_t shift = (state << 1 | taken) << 1;
    return (LUT >> shift) & 0x3;
  }

public:
  gshare(uint32_t entries, uint32_t hbits) :
    addr_bits(entries),
    history_bits(hbits),
    table_size(1 << entries) {
    uint32_t table_words = table_size / 16;
    table = new uint32_t[table_words];
    for (uint32_t i = 0; i < table_words; i++) {
      // initialize to weakly not taken
//      table[i] = 0x55555555;
      // initialize to weakly taken
      table[i] = 0xaaaaaaaa;
    }
  }

  ~gshare() {
    delete [] table;
  }

  bool predict(uint32_t addr, long_vector_c& global_history) {
    uint32_t hval  = get_hval(addr, global_history);
    uint32_t state = get_table_entry(hval);
    bool result = (state >> 1); // high bit is prediction
    return result;
  }

  void update(const uint32_t& addr, const long_vector_c& global_history, const bool& taken) {
    uint32_t hval = get_hval(addr, global_history);
    uint32_t state = get_table_entry(hval);
    set_table_entry(hval, next_state(state, taken));
  }
};
                                                                                                              
#endif /* GSHARE_H */
