#ifndef ELF_LOADER_H
#define ELF_LOADER_H

#include <inttypes.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

#include <assert.h>

#include <map>

#include "elf32.h"
#include "sim_endian.h"

// memcpy_from_host should be a functor with the syntax and semantics
// of the stdc memcpy procedure.
template <class memcpy_from_host_t>
class elf_loader
{
public:
  typedef struct elf_loader_return_values {
    uint32_t start_pc;
    uint32_t start_freemem;
    uint32_t start_segment;
    uint32_t end_segment;
  } elf_return;

  typedef map < uint32_t, uint32_t > text_sym_map_t; // map from vaddr -> string table offset


private:
  int RawCheckHeader(Elf32_Ehdr *hdr)
    {
      if(!(IS_ELF(*hdr)) ||
         hdr->e_ident[EI_CLASS] != ELFCLASS32 ||
         hdr->e_ident[EI_DATA] != ELFDATA2LSB ||
         hdr->e_ident[EI_VERSION] != EV_CURRENT ||
         CVT_ENDIAN_HWORD(hdr->e_machine) != EM_RAW)
      {
        return 1;
      }

      return 0;
    }

  int RawCheckIsExecutable(Elf32_Ehdr *hdr)
    {
      return (CVT_ENDIAN_HWORD(hdr->e_type) != ET_EXEC);
    }



  /*
    pread() reads up to count bytes from file descriptor fd at offset
    offset (from the start of the file) into the buffer starting at
    buf.  The file offset is not changed.  Contributed by Mark Dykstra
    2004-Sep-03
  */
  static
  ssize_t elf_pread(int fd, void *buf, size_t count, off_t offset) {
    off_t returned_offset;
    if ((returned_offset = lseek(fd, offset, SEEK_SET)) == -1) return(-1);
    ssize_t bytes_read = read(fd, buf, count);
    lseek(fd, returned_offset, SEEK_SET);
    return bytes_read;
  }

  int
  RawLoadExecutable(Elf32_Ehdr *ehdr,
                    int fd,
                    memcpy_from_host_t* memcpy_from_host,
                    elf_return* eret,
                    bool verbose)
    {
      uint16_t num_segments = CVT_ENDIAN_HWORD(ehdr->e_phnum);
      uint16_t header_size = CVT_ENDIAN_HWORD(ehdr->e_phentsize);
      uint32_t entry_pt = CVT_ENDIAN_WORD(ehdr->e_entry);
      uint32_t prog_header_offset = CVT_ENDIAN_WORD(ehdr->e_phoff);
      uint32_t total_mem = 0;

      eret->start_pc = entry_pt;

      for (int i = 0; i < num_segments; i++)
      {
        Elf32_Phdr phdr;
        /* read the ith program header ("program" headers give information about
           segments) */
        if (elf_pread(fd,
                      &phdr,
                      sizeof(Elf32_Phdr),
                      prog_header_offset + (i * header_size)) == -1) {
          return -1;
        }

        uint32_t seg_type = CVT_ENDIAN_WORD(phdr.p_type);
        uint32_t seg_vaddr = CVT_ENDIAN_WORD(phdr.p_vaddr);
        uint32_t seg_memsz = CVT_ENDIAN_WORD(phdr.p_memsz);
        uint32_t seg_flags = CVT_ENDIAN_WORD(phdr.p_flags);
        uint32_t seg_fileOffset = CVT_ENDIAN_WORD(phdr.p_offset);
        uint32_t seg_fileSize = CVT_ENDIAN_WORD(phdr.p_filesz);
        uint32_t mem_end = seg_vaddr + seg_memsz;

        if (mem_end > total_mem) {
	  total_mem = mem_end;
	}

        if ((seg_type & PT_LOAD) && (seg_fileSize > 0))
        {
          uint8_t* membuf = new uint8_t[seg_fileSize];

          printf("loading segment: start 0x%08x, size 0x%08x, permissions %c%c%c\n",
                 seg_vaddr, seg_memsz,
                 ((seg_flags & PF_R) ? 'r' : '-'),
                 ((seg_flags & PF_W) ? 'w' : '-'),
                 ((seg_flags & PF_X) ? 'x' : '-')
            );

          // read data directly into membuf, without converting for
          // endianness, endianness conversion is handled by the mem
          // load/store routines.
          if (elf_pread(fd, membuf, seg_fileSize, seg_fileOffset) == -1) return -1;
          // Copy into simulator.  We use seg_fileSize instead of
          // seg_memsz, because the simulator does allocate on write, and
          // any data that isn't in the file is bogus anyway (I hope).
          memcpy_from_host->operator()(seg_vaddr, membuf, seg_fileSize);
          delete[] membuf;
        }

      }

      printf("Entry point 0x%08x\n",
             entry_pt);

      // now tell the application where the free memory starts:
      eret->start_freemem = (total_mem + 16) & 0xfffffff0;

      {
        uint16_t num_sections = CVT_ENDIAN_HWORD(ehdr->e_shnum);
        uint16_t header_size = CVT_ENDIAN_HWORD(ehdr->e_shentsize);
        uint32_t section_header_offset = CVT_ENDIAN_WORD(ehdr->e_shoff);

        for (int i = 0; i < num_sections; i++)
        {
          Elf32_Shdr shdr;
          /* read the ith section header */
          elf_pread(fd, &shdr, sizeof(Elf32_Shdr), section_header_offset + (i * header_size));

          if (((shdr.sh_flags & SHF_EXECINSTR) != 0) &&
              (shdr.sh_type == SHT_PROGBITS)) {
            printf(".text is from 0x%08x to 0x%08x\n",
                   shdr.sh_addr, shdr.sh_addr + shdr.sh_size);
            eret->start_segment = shdr.sh_addr;
            eret->end_segment = shdr.sh_addr + shdr.sh_size;
            break;
          }
        }
      }

      return 0;
    }

  int
  LoadSymtab(Elf32_Ehdr *ehdr,
             int fd,
             char **string_table,
             text_sym_map_t* sym_map)
    {
      uint16_t num_sections = CVT_ENDIAN_HWORD(ehdr->e_shnum);
      uint16_t header_size = CVT_ENDIAN_HWORD(ehdr->e_shentsize);
      uint32_t section_header_offset = CVT_ENDIAN_WORD(ehdr->e_shoff);
      int string_section = -1;

      for (int i = 0; i < num_sections; i++)
      {
        Elf32_Shdr shdr;
        /* read the ith section header */
        elf_pread(fd, &shdr, sizeof(Elf32_Shdr), section_header_offset + (i * header_size));

        if (shdr.sh_type == SHT_SYMTAB) {
          // symbol table:
          assert (shdr.sh_entsize == sizeof(Elf32_Sym));
          assert (shdr.sh_link != 0);
          string_section = shdr.sh_link;
          size_t num_entries = shdr.sh_size / shdr.sh_entsize;
          printf("%d symbol table entries\n", num_entries);
          Elf32_Sym *symtab = new Elf32_Sym[num_entries];
          elf_pread(fd, symtab, shdr.sh_size, shdr.sh_offset);

          for (size_t jj = 0; jj != num_entries; jj++) {
            if ((symtab[jj].st_shndx == 1) && (symtab[jj].st_name != 0)) {
              (*sym_map)[symtab[jj].st_value] = symtab[jj].st_name;
            }
          }
          break;
        }
      }

      {
        // now read the string table:
        assert(string_section >= 0);
        Elf32_Shdr shdr;
        elf_pread(fd,
                  &shdr,
                  sizeof(Elf32_Shdr),
                  section_header_offset + (string_section * header_size));

        assert(shdr.sh_type == SHT_STRTAB);

        *string_table = new char[shdr.sh_size];

        elf_pread(fd, *string_table, shdr.sh_size, shdr.sh_offset);
      }
      
      return 0;
    }

public:
  elf_loader() { }

  int
  read_elf_file(const char* filename,
                memcpy_from_host_t* memcpy_from_host,
                elf_return *eret,
                bool verbose) {
    Elf32_Ehdr ehdr;
    int fd;
    
    if ((fd = open(filename, O_RDONLY, 0)) < 0)
    {
      fprintf(stderr, "couldn't open %s\n", filename);
      return 1;
    }

    if (elf_pread(fd, &ehdr, sizeof(Elf32_Ehdr), 0) != sizeof(Elf32_Ehdr))
    {
      fprintf(stderr, "file %s: couldn't read ELF header\n", filename);
      return 1;
    }

    if (RawCheckHeader(&ehdr) != 0)
    {
      printf("%x %x %x\n",
             CVT_ENDIAN_HWORD(ehdr.e_machine),
             CVT_ENDIAN_HWORD(ehdr.e_type),
             ehdr.e_ident[EI_DATA]);
      fprintf(stderr, "file %s: not a proper header\n", filename);
      return 1;
    }

    if (RawCheckIsExecutable(&ehdr) != 0)
    {
      fprintf(stderr, "file %s: not a raw executable\n", filename);
      return 1;
    }

    if (RawLoadExecutable(&ehdr, fd, memcpy_from_host, eret, verbose) != 0)
    {
      fprintf(stderr, "file %s: problem reading executable\n", filename);
      return 1;
    }

    close(fd);

    return 0;
  }

  int
  read_symtab(const char* filename,
              char **string_table,
              text_sym_map_t* sym_map) {
    Elf32_Ehdr ehdr;
    int fd;
    
    if ((fd = open(filename, O_RDONLY, 0)) < 0)
    {
      fprintf(stderr, "couldn't open %s\n", filename);
      return 1;
    }

    if (elf_pread(fd, &ehdr, sizeof(Elf32_Ehdr), 0) != sizeof(Elf32_Ehdr))
    {
      fprintf(stderr, "file %s: couldn't read ELF header\n", filename);
      return 1;
    }

    if (RawCheckHeader(&ehdr) != 0)
    {
      printf("%x %x %x\n",
             CVT_ENDIAN_HWORD(ehdr.e_machine),
             CVT_ENDIAN_HWORD(ehdr.e_type),
             ehdr.e_ident[EI_DATA]);
      fprintf(stderr, "file %s: not a proper header\n", filename);
      return 1;
    }

    if (RawCheckIsExecutable(&ehdr) != 0)
    {
      fprintf(stderr, "file %s: not a raw executable\n", filename);
      return 1;
    }

    if (LoadSymtab(&ehdr, fd, string_table, sym_map) != 0)
    {
      fprintf(stderr, "file %s: problem reading executable\n", filename);
      return 1;
    }

    close(fd);

    return 0;
  }
};

#endif /* ELF_LOADER_H */
