#ifndef DELAY_WRITE_H
#define DELAY_WRITE_H

#include <inttypes.h>
#include "circuit.h"
#include <map>
#include <ext/hash_set>
#include "globals.h"

using namespace std;

extern uint64_t cycle_count;


template <class T>
class result_bus : clocked_net, public port<const T>
{
private:
  enum { MAX_DELAY = 10000 * 8 };
  typedef map<uint64_t, T> time_q_t;
  typedef typename time_q_t::iterator time_q_iterator_t;
  time_q_t data;
  uint64_t time;
  T output;

  // do_pulse runs at the beginning of each clock cycle
  void do_pulse() {
    if (!data.empty() ) {
      time_q_iterator_t i = data.begin();
      assert(i->first >= time);
      if ( i->first == time) {
        output         = i->second;
        data.erase(i);
      } else output = T();
    } else {
      output = T();
    }
    time = (time+1);
  }
  
public:
  result_bus() :
    clocked_net(),
    data(),
    time(0),
    output() { }

    inline
    void write(const T& v, size_t delay = 0) {
      if (delay >= MAX_DELAY) printf("ASSERTION FAILURE AT CYCLE %llu\n", cycle_count);
      assert(delay < MAX_DELAY);
      assert( !is_in_use(delay)); // there should never be two people driving
      uint64_t entry_num = (time + delay);
      data[entry_num] = v;
    }

  // SMT: per-context flush (cancel everything in fifo that matches the thread_id)
  // 2005.01.28-woley-assumes that <class T> has a member "context":
  //   this should be implemented in a non-templitized bus type
  inline
  void flush(uint32_t context, uint64_t timestamp) {
    time_q_iterator_t i = data.begin();
    while(i != data.end()) {
      if ((i->second.context == context) && 
          (i->second.instr_num >= timestamp)) {
        data.erase(i++);
      }
      else {
        ++i;
      }
    }
  }

  inline
  bool is_in_use(uint64_t delay = 0) {
    return (data.find( time+delay ) != data.end());
  }
  
  const T& operator()() const { return output; }
};

#endif /* DELAY_WRITE_H */
