#ifndef CACHE_H_GUARD
#define CACHE_H_GUARD

#include <vector>
#include <inttypes.h>

extern uint32_t debug_mask;
extern uint64_t cycle_count;

using namespace std;

//for finding lg of cache line size, the value of the 
//lg should be passed to the cache in initialization
inline uint32_t lg(uint32_t in) {
  uint32_t temp = 0;
  while (!(in & 0x1)) {
    temp++;
    in = in >> 1;
  }
  // insure in is a power of 2
  assert(in == 1);
  return temp;
}

template <class DATA> class cache {
private:
  typedef vector<DATA>     cacheline_t;
  typedef vector<uint32_t> tagline_t;
  typedef vector<uint32_t> lruline_t;

  vector<cacheline_t> cache_data;
  vector<tagline_t>   cache_tags;
  vector<lruline_t>   cache_lru;
  uint32_t lines;
  uint32_t assoc;
  uint32_t line_size; //in bytes
  uint32_t lg_line_size; //easier than calculating

  // changes lru values based on a cache entry-way reference
  void lru_ref(uint32_t line, uint32_t way) { 
    uint32_t old_way_lru_val = cache_lru[line][way];
    
    for(uint32_t i = 0; i < assoc; i++) {
      if(i == way)
        cache_lru[line][i] = 0;
      else if(cache_lru[line][i] < old_way_lru_val)
        cache_lru[line][i]++;
    }
  }
  
  // finds the lru way
  uint32_t lru_lookup(uint32_t line) {
    uint32_t least_rec_used_val = assoc - 1;

    for(uint32_t i = 0; i < assoc; i++) {
      if(cache_lru[line][i] == least_rec_used_val) {
        return i;
      }
    }

    assert(true && "lru_lookup failed!");
    return assoc;
  }


  // evicts the lru element on the specified line and returns the 'way' of the evicted element
  uint32_t lru_evict(uint32_t line) {
    uint32_t least_rec_used_way = 0;
    uint32_t least_rec_used_val = assoc - 1;

    for(uint32_t i = 0; i < assoc; i++) {
      if(cache_lru[line][i] == least_rec_used_val) {
        cache_lru[line][i] = 0;
        least_rec_used_way = i;
      }
      else {
        cache_lru[line][i]++;
      }
    }

    //    if(debug_mask & 0x0001) printf("CACHE> lru evict: line %d way %d\n", line, least_rec_used_way);
    
    return least_rec_used_way;
  }
  
 public:
  
  cache(uint32_t num_lines, uint32_t line_size, uint32_t lg_line_size, uint32_t associativity) :
    cache_data(num_lines, cacheline_t(associativity)),
    cache_tags(num_lines, tagline_t(associativity, 0)),
    cache_lru(num_lines,  lruline_t(associativity,0)),
    lines(num_lines),
    assoc(associativity),
    line_size(line_size),
    lg_line_size(lg_line_size)
    { 
      // initialize the LRU and tag entries
      for(uint32_t line = 0; line < num_lines; line++) {
        for(uint32_t way = 0; way < assoc; way++) {
          cache_tags[line][way] = 0xeaddead; //initialize to dead value
          cache_lru[line][way] = way;
        }
      }
    }
  
  inline uint32_t get_index(uint32_t addr) {
    return (addr >> lg_line_size) & (lines - 1);
  }

  inline uint32_t get_tag(uint32_t addr) {
    return (addr & ~((line_size * lines) - 1));
  }
  
  // checks for a cache hit. on a hit, sets way and returns true. on a miss, sets way to assoc and returns false.
  bool check(uint32_t addr, uint32_t& way) {
    // hash function
    uint32_t line = get_index(addr);
    uint32_t tag = get_tag(addr);

    //    if(debug_mask & 0x0001) printf("CACHE> Check addr: %x tag: %x line: %d", addr, tag, line);

    for(way = 0; way < assoc; way++) {
      if(tag == cache_tags[line][way]) {
        //        if(debug_mask & 0x0001) printf(" way: %d\n", way);
        return true;
      }
    }
    //    if(debug_mask & 0x0001) printf(" addr not found\n");
    return false;
  }
  
  // the user should call check before calling get_value
  DATA get_value(uint32_t addr, uint32_t way) {
    // hash function
    uint32_t line = get_index(addr);
    
    lru_ref(line, way);
    return cache_data[line][way];
  }
  
  void update_value(uint32_t addr, DATA value) {
    // hash function
    uint32_t line = get_index(addr);
    
    uint32_t way = 0;
    bool tag_check = check(addr, way);
    if(tag_check) 
      lru_ref(line, way);
    else
      way = lru_evict(line);
    
    cache_tags[line][way] = get_tag(addr);
    cache_data[line][way] = value;
  }

  // note:  this function is specialized for timestamp data updates
  void conditional_update_value(uint32_t addr, uint64_t value, uint32_t latency) {
    // hash function
    uint32_t line = get_index(addr);
    
    uint32_t way = 0;
    bool tag_check = check(addr, way);
    if(tag_check) 
      lru_ref(line, way);
    else {
      // do not update the cache line/way if a previous value
      // is larger than the cycle_count:  useful when the DATA
      // value represents a timestamp, as we do not want to
      // evict a cache line that has not yet been read from

      // note:  the latency is to allow time for the reader
      // grab the value since the cache does not notify
      // when a value is ready
      uint64_t line_ts = cache_data[line][lru_lookup(line)];
      if(cycle_count < line_ts + latency) return;
      
      way = lru_evict(line);
      //      if(debug_mask & 0x0001) printf("CACHE> conditional_update_value: cache_data[%d][%d]: %llu value: %llu\n", line, way, cache_data[line][way], value);

    }
    cache_tags[line][way] = get_tag(addr);
    cache_data[line][way] = value;
  }

};

#endif /* CACHE_H_GUARD */
