//Testing SVN with this comment
#include <stdio.h>
#include <stdlib.h>
#include <inttypes.h>
#include <vector>
#include <map>
#include <string.h>

#include "bitops.h"
#include "sparse_memory.h"
#include "decode.h"
#include "execute.h"
#include "arch-model.h"
#include "arch-listeners.h"

char *cfg_file = NULL;
bool make_cfg = false;

using namespace std;
bool trace_on = false;
bool turnoff = false;
uint32_t gc_address = 0;
uint32_t stack_limit = 0;
uint32_t debug_mask = 0;
uint32_t lowest_sp = 0xffffffff;
int garbage_collected = 0;
map <uint32_t, bool> miss_skip_map;

bool    use_spawn_predictor = 0; // default off until tested
uint32_t sp_req_correct = 8;
bool     sp_perfect = true;
uint32_t sp_size = 4 * 1024;
uint32_t sp_assoc = 4;
uint32_t sp_window_size = 4000;
uint32_t sp_max_open_entries = 0;
uint32_t sp_close_early = 0;


// DEFAULT SETTINGS FOR TIMING-MODEL COMMAND OPTIONS
bool stack_cache_version = false;
uint32_t superpipeline_factor = 1;
bool br_pred_disabled = false;
uint32_t iL1_lsize = 128;
uint32_t iL1_assoc = 1;
uint32_t iL1_lines = 256;
uint32_t iL1_delay = 10;
uint32_t dL1_lsize = 64;
uint32_t dL1_assoc = 4;
uint32_t dL1_lines = 32;
uint32_t dL1_delay = 10;
uint32_t L2_lsize = 128;
uint32_t L2_assoc = 8;
uint32_t L2_lines = 512;
uint32_t L2_delay = 100;
uint32_t iwin_size = 0xFFFFFFFF;
uint32_t bpred_ltblsize = 1024;
uint32_t bpred_gtblsize = 4096;
uint32_t bpred_ctblsize = 4096;
char* flowgraph_file;
bool spcdmf_mode = false;
bool disable_single_issue;
bool str_pfetch_enabled = false;
bool str_pfetch_l2only = false;
uint64_t debug_start_instr = 0;
uint64_t debug_num_instrs = 0;
uint64_t num_instrs = 0;
bool debug_enabled = true;


// Usage/Help function called by command line decoding for statement
void
usage(void) 
{
  printf("\n\tARCHITECTURE SIMULATOR with fast Timing Model\n");
  printf("\n\tDeveloped by: Professor Matthew Frank and his students\n");
  printf("\tUniversity of Illinois at Urbana-Champaign\n");
  printf("\nUsage:\n\tarch-model [Options ...] exec_name args\n");
  printf("\nOptions:  (In any order; \"-tm:\" indictates timing-model sepecific option)\n");
  printf("\t-help,-h\tUsage text\n");
  printf("\t-trace\t\tOutputs the PC for every instruction\n\t\t\texecuted in order to compare with arch-model output\n");
  printf("\t-skip <n,n>\tCreates a skip mapping so you can skip a load miss\n");
  printf("\t-debug <mask>\tUse bit mask, mask[31:0], to turn on debugging output:\n");
  printf("\t\t\tNOTE: A default setting of all zeroes gives no ouput\n");
  printf("\t\t\t      and n is a decimal number input\n");
  printf("\t  bit#\t0.\tDebug info for timing model\n");
  printf("\t-proto\tBare bones simulator shell for fast prototyping\n");
  printf("\t-depfree\tTurns on the depfree-model with default options\n");
  printf("\t-depfree:mode:<option>\n");
  printf("\t Type:\tbase\tLam & Wilson \"base\" model\n");
  printf("\t\tsp\tLam & Wilson \"sp\" model, DEFAULT\n");
  printf("\t\tspcdmf <filename>  Lam & Wilson \"sp-cd-mf\" model,\n\t\t\tonly for the spcdmf mode do you need to specify a\n\t\t\tfilename\n");
  printf("\t-tm\t\tTurns on the timing-model with default options\n");
  printf("\t-tm:bpred_off\tTurns off branch predictor and jump penalty\n");
  printf("\t-tm:infissue\tAllows for infinite issue width, DEFAULT=single\n");
  printf("\t-tm:pfetch\tEnables stream prefetcher, both L1 and L2\n");
  printf("\t-tm:pfetchl2\tEnalbes only stream prefetching into the L2\n");
  printf("\t-dbgstart n\tNumber of instrs to wait before printing debug info\n");
  printf("\t-dbginstrs n\tNumber of instrs to print debug info for\n");
  printf("\t-tm:sp <n>\tTurns on super pipeline factor that multiplies\n\t\t\tthe ADD latency by n\n");
  printf("\t-tm:iwin <n>\tSets instruction window size to n, default = 16\n");
  printf("\t-tm:scache\tTurns on stack cache simulation\n");
  printf("\t-tm:cpfile <name>  Turns on crit path data generator and writes to file\n");
  printf("\t-tm:bpred:<type>:<option> <n>\n");
  printf("\t TYPE:\ttournament\tTournament predictor style\n");
  printf("\t Option:ltblsize\tNumber of entries in local history table\n");
  printf("\t\tgtblsize\tNumber of entries in global history table\n");
  printf("\t\tctblsize\tNumber of entries in chooser table\n");
  printf("\t-tm:cache:<type>:<option> <n>\n");
  printf("\t Type:\tiL1\tInstruction L1 cache\n\t\tdL1\tData L1 cache\n");
  printf("\t\tL2\tL2 cache\n");
  printf("\t Option:lsize\tLine size in bytes\n");
  printf("\t\tassoc\tSet associativity of the cache\n");
  printf("\t\tlines\tNumber of entries or lines in the cache\n");
  printf("\t\tdelay\tNumber of delay cycles to next higher level of memory\n");
  printf("\nRequired:  (Must go last)\n");
  printf("\texec_name\tFile name of executeable to be run on simulator\n");
  printf("\targs\t\tThe arguements to be inputed into simulated program\n");
  printf("\nExamples:\n\tExample of using debug mask with full debug info using timing-model:\n");
  printf("\t\tDebug mask = 1 in binary or 0x1 in hex, enter number in decimal\n");
  printf("\t\tarch-model -tm -debug 1 lzw 5\n");
  printf("\n");
  exit(0);
}

void input_error(char* error_input, char* error_option) {
  printf("\nERROR: the value %s is invalid input for the option \"%s\"\n",
         error_input, error_option);
  usage();

  exit(0);
}

void prog_comm_error(char* error_option) {
  printf("\nERROR: You have entered the option \"%s\" after already entering the\n",
         error_option);
  printf("\tsimulation program name\n");
  usage();

  exit(0);
}

int
main(int argc, char* argv[])
{
  sparse_memory *the_mem = new sparse_memory();
  arch_model model(the_mem, true);
  track_last_writer* the_last_writer_tracker = new track_last_writer(&model);
  new track_call_depth(&model);
  gen_cfg* the_cfg_generator = 0;
  int prog_argc = 0;
  char* cache_type;
  char* cache_option;
  uint32_t stop_inst = 0;

  model.register_last_writer_tracker(the_last_writer_tracker);

  // **** PRINTS USAGE IF NO ARGS ARE FOUND WITH PROGRAM
  if(argc == 1)
    usage();

  // **** COMMAND LINE DECODE
  for (int i = 1; i < argc; i++) {
    // Help and usage output
    if(!strcmp(argv[i], "-help") || !strcmp(argv[i], "-h"))
      usage();
    // Trace mode prints out the PC to compare with arch-model program flow
    else if (!strcmp(argv[i], "-stop")) {
      i++;
      stop_inst = strtoul(argv[i], 0, 0);
    }
    else if (!strncmp(argv[i], "-trace", 7))
    {
      trace_on = true;
    }
    // Provides debug information as set by number based on which information
    // you would like to see
    else if (!strncmp(argv[i], "-debug", 7))
    {
      i++;
      debug_mask = strtoul(argv[i], 0, 0);
    }
    else if (!strcmp(argv[i], "-skip"))
    {
      char* tokenPtr;

      i++;
      tokenPtr = strtok(argv[i], ",");

      while(tokenPtr != NULL) {
        miss_skip_map[strtoul(tokenPtr, 0, 0)] = true;
        tokenPtr = strtok(NULL, ",");
      }
      printf("Size of skip map: %u\n", miss_skip_map.size());
    }
    else if (!strncmp(argv[i], "-stack_limit", 13))
    {
      i++;
      stack_limit = strtoul(argv[i], 0, 0);
    }
    else if (!strncmp(argv[i], "-tm:sp", 6)) 
    {
      i++;
      superpipeline_factor = strtoul(argv[i], 0, 0);
    }
    else if (!strncmp(argv[i], "-tm:bpred_off", 10)) 
    {
      br_pred_disabled = true;
    }
    else if (!strncmp(argv[i], "-tm:iwin", 8)) 
    {
      i++;
      iwin_size = strtoul(argv[i],0,0);
    }
    else if (!strncmp(argv[i], "-tm:bpred:tournament:ltblsize", 8))
    {
      i++;
      bpred_ltblsize = strtoul(argv[i],0,0);
    }
    else if (!strncmp(argv[i], "-tm:bpred:tournament:gtblsize", 8)) 
    {
      i++;
      bpred_gtblsize = strtoul(argv[i],0,0);
    }
    else if (!strncmp(argv[i], "-tm:bpred:tournament:ctblsize", 8)) 
    {
      i++;
      bpred_ctblsize = strtoul(argv[i],0,0);
    }
    else if (!strcmp(argv[i], "-tm:infissue"))
    {
      disable_single_issue = true;
    }
    else if (!strcmp(argv[i], "-tm:pfetch"))
    {
      str_pfetch_enabled = true;
    }
    else if (!strcmp(argv[i], "-tm:pfetchl2"))
    {
      str_pfetch_enabled = true;
      str_pfetch_l2only = true;
    }
    else if (!strcmp(argv[i], "-dbgstart"))
    {
      i++;
      debug_start_instr = strtoull(argv[i],0,0);
    }
    else if (!strcmp(argv[i], "-dbginstrs"))
    {
      i++;
      debug_num_instrs = strtoull(argv[i],0,0);
    }
    else if (!strncmp(argv[i], "-tm:cache:", 10)) 
    {
      cache_type = strtok(argv[i], ":");
      cache_type = strtok(NULL, ":");
      cache_type = strtok(NULL, ":");
      cache_option = strtok(NULL, ":");

      i++;
      if(!strcmp(cache_type, "iL1")) {
        if(!strcmp(cache_option, "lsize"))
          iL1_lsize = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "assoc"))
          iL1_assoc = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "lines"))
          iL1_lines = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "delay"))
          iL1_delay = strtoul(argv[i],0,0);
        else 
          input_error(argv[i-1], argv[i]);
      }
      else if(!strcmp(cache_type, "dL1")) {
        if(!strcmp(cache_option, "lsize"))
          dL1_lsize = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "assoc"))
          dL1_assoc = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "lines"))
          dL1_lines = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "delay"))
          dL1_delay = strtoul(argv[i],0,0);
        else 
          input_error(argv[i-1], argv[i]);
      }
      else if(!strcmp(cache_type, "L2")) {
        if(!strcmp(cache_option, "lsize"))
          L2_lsize = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "assoc"))
          L2_assoc = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "lines"))
          L2_lines = strtoul(argv[i],0,0);
        else if(!strcmp(cache_option, "delay"))
          L2_delay = strtoul(argv[i],0,0);
        else 
          input_error(argv[i-1], argv[i]);
      }
      else
        input_error(argv[i-1], argv[i]);
    } 
    else if (!strcmp(argv[i], "-tm"))
    {
    }
    else if (!strcmp(argv[i], "-depfree:mode:base"))
    {
      br_pred_disabled = true;
    }
    else if (!strcmp(argv[i], "-depfree:mode:sp"))
    {
    }
    else if (!strcmp(argv[i], "-depfree:mode:spcdmf"))
    {
      i++;
      spcdmf_mode = true;
      flowgraph_file = argv[i];
    }
    else if (!strcmp(argv[i], "-depfree"))
    {
    }
    else if(!strcmp(argv[i], "-use_spawn_pred")) {
      use_spawn_predictor = true;
    }
    else if(!strcmp(argv[i], "-sp_perfect")) {
      i++;
      sp_perfect = atoi(argv[i]);
    }
    else if(!strcmp(argv[i], "-sp_max_open")) {
      i++;
      sp_max_open_entries = atoi(argv[i]);
    }
    else if(!strcmp(argv[i], "-sp_req_correct")) {
      i++;
      sp_req_correct = atoi(argv[i]);
    }
    else if(!strcmp(argv[i], "-sp_max_open")) {
      i++;
      sp_max_open_entries = atoi(argv[i]);
    }
    else if(!strcmp(argv[i], "-sp_close_early")) {
      i++;
      sp_close_early = atoi(argv[i]);
    }
    else if(!strcmp(argv[i], "-make_cfg")) {
      i++;
      make_cfg = true;
      cfg_file = argv[i];
    }
    else if(!strcmp(argv[i], "-cfg_file")) {
      i++;
      cfg_file = argv[i];
    }
    else
    {
      prog_argc = i;
      break;
    }
  }

  printf("POLYFLOW ISA SIMULATOR\n");
  printf("University of Illinois at Urbana-Champaign\n");

  // **** LOADING TEXT SECTION OF ELF EXECUTABLE
  datapath_reg_init dri;

  if (model.load_elf_file(argv[prog_argc], argc-prog_argc, &argv[prog_argc], &dri) != 0) {
    fprintf(stderr, "arch_model: error loading program executable file %s!\n", argv[prog_argc]);
    exit(1);
  }

  model.init_regs(&dri);

  if (make_cfg) {
    // attach a cfg generator to the arch model:
    printf("starting cfg generator\n");
    the_cfg_generator = new gen_cfg(&model, dri.pc);
  }

  // **** STEPPING THE SIMULATOR INSTR BY INSTR
    uint64_t the_instr_count = 0;
    FILE* tracefile = 0;

    if (trace_on) {
      tracefile = fopen("trace.out", "w");
      if (tracefile == 0) {
        fprintf(stderr, "couldn't open trace.out\n");
        exit(1);
      }
      // create a state spewer listener:
      new spew_state_info(&model, tracefile);
    }

    FILE* spawn_pts = 0;
    spawn_pts = fopen("spawnpts.out", "w");
    if(spawn_pts == 0) {
      fprintf(stderr, "could not open spawnpts.out");
      exit(1);
    }
    fprintf(spawn_pts, "SPAWNPT\tFN\tINSTRS\n");

    while (!model.halted) {

      const decoder::instr_h& the_instr = model.do_cycle(tracefile);

      //to prevent a warning
      if(0)
        printf("%u \n", the_instr->context);

      the_instr_count++;

      if(stop_inst && (the_instr_count >= stop_inst)) break;
    }
    printf("arch_model::"); model.print_sparse_memory_stats();
    printf("Instuctions executed: %llu\n", the_instr_count);

  if (the_cfg_generator != 0) {
    printf("writing cfg file\n");
    FILE* cfgf = fopen(cfg_file, "w");
    if (cfgf == 0) {
      fprintf(stderr, "ERROR: couldn't open cfg output file %s for write\n", cfg_file);
    }
    else {
      the_cfg_generator->print(cfgf);
      fclose(cfgf);
    }
  }

}
