#ifndef POOL_ALLOC_H
#define POOL_ALLOC_H

#include <vector>
#include <iostream>
#include "free_list.h"
#include "globals.h"

//NOTE: We're passing back pointers to vector elements, for efficiency. Since there's no resizing going on,
//it should be okay. Just need to remember that the pointers returned by get_element() and pop_head()
//should not be stored anywhere. 


extern uint32_t num_contexts; //number of contexts is a global variable in the simulator. 

using namespace std;


//Remember that you MUST provide a default constructor for class C. AND
//the default constructor should, ideally, create a structure with its 'valid' bit set to false.


//policy decision: there's a common pool, from which a number of different contexts
//want to allocate slots. Contexts complete with each other. To guarantee forward progress,
//there should be high-watermarks on the number of slots a particular context can occupy.
//Moreover, these high-watermarks depend on how speculative a context is. EG: The non-speculative
//context should probably get all the ROB slots that it asks for. 


//C is the datatype you want to store in the pool allocator. 
//head and tail both point to VALID entries!
template <class C>
class pool_alloc {

 private:

  class element {         
   public:
    C data;
    uint32_t next;
    uint32_t prev;
    element():next(INV_POOL_PTR), prev(INV_POOL_PTR) {}  //default constructor so we can make vectors from this
    element(C x_data): data(x_data), next(INV_POOL_PTR), prev(INV_POOL_PTR) {}
  };

  uint32_t pool_size;          //total size of the pool

  vector<element> array;     //vector which stores the linked list

  free_list FL;

  vector<uint32_t> heads;    //one per context, head and tail pointers.
  vector<uint32_t> tails;   
  vector<uint32_t> counts;   //number of slots occupied by a context

  vector<uint32_t> max_counts; //per-context count of the maximum slots that can be occupied by a particular context. 

 public:

  pool_alloc(uint32_t x_pool_size, vector<uint32_t>  x_max_counts);  //constructor
  pool_alloc(uint32_t x_pool_size); //constructor. When using this constructor, max_size PER CONTEXT = pool size

  void set_max_counts(vector<uint32_t> x_max_counts);   //manually set the max size per context

  C* get_element(uint32_t slot);         //returns a pointer to the data. 
  C& operator[](uint32_t slot);

  uint32_t push_tail(uint32_t context, C value);  //insert an element at the tail of a context. Returns slot if insertion succeeds; returns INV_POOL_PTR if insertion fails

  bool pop_head(uint32_t context, C** value);   //delete element from head of a context, and return a pointer to the deleted element (element is still in the pool)
  //done this way so the user can, eg, invalidate data in an element once its been deleted. Returns success/failure

  bool pop_tail(uint32_t context, C** value);  //delete an element from the TAIL of a context. Needed when doing partial flushing

  uint32_t num_entries(uint32_t context);  //get the number of slots occupied by a context
  uint32_t num_entries();                  //total number of entries

  uint32_t get_pool_size();                    //returns pool size


  uint32_t num_free_slots();               //total number of free slots
  uint32_t num_free_slots_context(uint32_t context);    //number of free slots for a particular context

  bool is_empty(uint32_t context);                      //does the pool have any entries from a particular context

  uint32_t get_next(uint32_t slot);      //get ptr to next element. returns array[slot].next
  uint32_t get_prev(uint32_t slot);      //get slot to prev element. returns array[slot].prev

  uint32_t head(uint32_t context);       //get the head ptr
  uint32_t tail(uint32_t context);       //get the tail ptr

  uint32_t invalid();                        //returns the null pointer. In our case, its INV_POOL_PTR, -1

  void clear();                      //completely clears the whole pool
  void clear_context(uint32_t c);              //clears out a particular context

  void check_for_leaks();                //checks if any pool slots have been lost

 private: //some functions used internally
  void set_next(uint32_t slot, uint32_t value);  //set next ptr.
  void set_prev(uint32_t slot, uint32_t value);  //set prev ptr.

};


template <class C>
pool_alloc<C>::pool_alloc(uint32_t x_pool_size, vector<uint32_t> x_max_counts) 
  : pool_size(x_pool_size), array(pool_size), FL(pool_size, 0), heads(num_contexts, INV_POOL_PTR), 
    tails(num_contexts, INV_POOL_PTR), counts(num_contexts, 0), max_counts(x_max_counts)
{
  assert(pool_size < 0x7FFFFFFF); //since we're using -1 as INV_POOL_PTR, can only have pools of size 2^31
}

template <class C>
pool_alloc<C>::pool_alloc(uint32_t x_pool_size) 
  : pool_size(x_pool_size), array(pool_size), FL(pool_size, 0), heads(num_contexts, INV_POOL_PTR), 
    tails(num_contexts, INV_POOL_PTR), counts(num_contexts, 0), max_counts(vector<uint32_t>(num_contexts, pool_size/num_contexts))
{
  assert(pool_size < 0x7FFFFFFF); //since we're using -1 as INV_POOL_PTR, can only have pools of size 2^31
}

template <class C>
void pool_alloc<C>::set_max_counts(vector<uint32_t> x_max_counts)
{
  assert(x_max_counts.size() == num_contexts);  
  max_counts = x_max_counts;
}


template <class C>
inline C* pool_alloc<C>::get_element(uint32_t slot) 
{
  assert(slot<pool_size);
  return &array[slot].data;
}

template <class C>
inline C& pool_alloc<C>::operator[](uint32_t slot)
{
  assert(slot<pool_size);
  return array[slot].data;
}

template <class C>
inline uint32_t pool_alloc<C>::push_tail(uint32_t context, C value)
{
  assert(context<num_contexts);
  if(counts[context] == max_counts[context]) //this context should not get any more slots
    return INV_POOL_PTR;
  if(FL.num_free_objs() == 0)                  //no more free slots
    return INV_POOL_PTR;        

  //its all good, do the insert
  uint32_t slot;
  FL.get_free_obj(slot);   //this function WRITES to slot
  element insert_this(value);

  if(counts[context] == 0){ //empty list
    assert(heads[context]==invalid());
    assert(tails[context]==invalid());
    heads[context] = slot;
  }
  uint32_t old_tail = tails[context];
  uint32_t new_tail = slot;
  insert_this.prev = old_tail;
  if(old_tail!=invalid())
    set_next(old_tail,slot);  //old_tail.next = slot
  tails[context] = new_tail;          //advance tail

  array[slot] = insert_this;
  counts[context]++;
  return slot;
}

template <class C>
inline bool pool_alloc<C>::pop_head(uint32_t context, C** value)  //**, since we want to change the value of the pointer
{
  assert(context<num_contexts);
  if(counts[context]==0)
    return false; //unable to delete

  if(counts[context] == 1) //will delete the last element. 
    tails[context] = INV_POOL_PTR; 

  uint32_t old_head = heads[context];
  uint32_t new_head = get_next(old_head);
  *value=get_element(old_head);
  heads[context] = new_head;  //advance head
  if(new_head!=invalid())
    set_prev(new_head, INV_POOL_PTR); //do this only if new_head != NULL
  set_next(old_head, INV_POOL_PTR);  //just to be sure
  counts[context]--; //
  FL.put_free_obj(old_head);
  return true;
}


template <class C>
inline bool pool_alloc<C>::pop_tail(uint32_t context, C** value)  //**, since we want to change the value of the pointer
{
  assert(context<num_contexts);
  if(counts[context]==0)
    return false; //unable to delete

  if(counts[context] == 1) //will delete the last element. 
    heads[context] = INV_POOL_PTR; 

  uint32_t old_tail = tails[context];
  uint32_t new_tail = get_prev(old_tail);
  *value=get_element(old_tail);
  tails[context] = new_tail;  //advance head
  if(new_tail != invalid())
    set_next(new_tail, INV_POOL_PTR);
  set_prev(old_tail, INV_POOL_PTR);  //just to be sure
  counts[context]--;
  FL.put_free_obj(old_tail);
  return true;
}


//the rest are Utility functions
template <class C>
inline uint32_t pool_alloc<C>::num_entries(uint32_t context)
{
  assert(context<num_contexts);
  //  return pool_size - FL.num_free_objs();
  return counts[context];
}


template <class C>
inline uint32_t pool_alloc<C>::num_entries()
{
  return pool_size - FL.num_free_objs();
}



template <class C>
inline void pool_alloc<C>::clear()
{
  for(uint32_t i=0; i<num_contexts; i++)
    clear_context(i);
}

template <class C>
inline void pool_alloc<C>::clear_context(uint32_t c)
{
  uint32_t begin_count = counts[c];
  for(uint32_t current=0; current<begin_count; current++) {
    C* temp;
    pop_head(c, &temp);
  }
}

template <class C>
inline uint32_t pool_alloc<C>::num_free_slots()
{
  return FL.num_free_objs();
}


template <class C>
inline uint32_t pool_alloc<C>::num_free_slots_context(uint32_t context)
{
  assert(context<num_contexts);
  return min(num_free_slots(), max_counts[context] - counts[context]);
}

template <class C>
inline bool pool_alloc<C>::is_empty(uint32_t context)
{
  return (!counts[context]);
}

template <class C>
inline uint32_t pool_alloc<C>::get_next(uint32_t slot) 
{
  assert(slot<pool_size);
  return array[slot].next;
}

template <class C>
inline uint32_t pool_alloc<C>::get_prev(uint32_t slot)
{
  assert(slot<pool_size);
  return array[slot].prev;
}

template <class C>
inline uint32_t pool_alloc<C>::head(uint32_t context)
{
  assert(context<num_contexts);
  return heads[context];
}


template <class C>
inline uint32_t pool_alloc<C>::tail(uint32_t context)
{
  assert(context<num_contexts);
  return tails[context];
}


template <class C>
inline uint32_t pool_alloc<C>::invalid()
{
  return INV_POOL_PTR;
}


template <class C>
inline uint32_t pool_alloc<C>::get_pool_size()
{
  return pool_size;
}


template <class C>
inline void pool_alloc<C>::check_for_leaks()
{
  vector<uint32_t> ref_counts(pool_size, 0);

  //mark slots occupied
  for(uint32_t c=0; c<num_contexts; c++) {
    //traverse context c
    for(uint32_t index=head(c); index!=invalid(); index=get_next(index)) {
      ref_counts[index]++;
    }
  }


  //mark slots in free list
  for (uint32_t fl_slot = 0; fl_slot < FL.queue.size(); fl_slot++)
    ref_counts[FL.queue[fl_slot]]++;

  //find if something is wrong
  for(uint32_t i=0; i<pool_size; i++) {
    if(ref_counts[i] == 0)
      printf("ROB Slot %u is Lost!\n", i);
    if(ref_counts[i] >1)
      printf("ROB Slot %u is present in duplicate locations!\n", i);
  }
}




//private functions
template <class C>
inline void pool_alloc<C>::set_next(uint32_t slot, uint32_t value)
{
  assert(slot<pool_size);
  array[slot].next = value;
}


template <class C>
inline void pool_alloc<C>::set_prev(uint32_t slot, uint32_t value)
{
  assert(slot<pool_size);
  array[slot].prev = value;
}

#endif


////////////Test Program and example of usage////////////////
//To compile this program, replace 'extern uint32_t num_contexts' in pool_alloc.h by 'uint32_t num_contexts'
//Then do g++ pool_alloc.cc free_list.cc
/*
class rob_struct
{
 public:
  uint32_t opcode;
  bool valid; 
  rob_struct(): valid(false) {}
  rob_struct(uint32_t value): opcode(value), valid(true) {}
};

int main()
{
  uint32_t rob_size;
  cout<<"Enter number of contexts: ";
  cin>>num_contexts;
  cout<<"\nEnter rob_size per context: ";
  cin>>rob_size;
  vector<uint32_t> max_counts(num_contexts, rob_size);  //simple 
  //Constructor example
  pool_alloc<rob_struct> my_rob(rob_size*num_contexts, max_counts); //creating a 512 entry ROB with 8 contexts

  bool quit = false;
  while(!quit) {
    uint32_t insert, context, temp_slot;
    rob_struct temp_rob_struct;
    rob_struct* ptr_struct;
    cout<<"Enter option:\n";
    cout<<"1:Insert,  2:Delete_head 3:Delete_tail  4:Print_all  5:Print_context  6:Print_all(reverse traversal)    ";
    cout<<"7:Show element  8:Modify-element  9.Clear_context  10.Clear_pool   11:Quit \n";
    uint32_t input;
    cin>>input;
    switch(input) {
    case 1:
      cout<<"\nEnter context: ";
      cin>>context;
      cout<<"Enter element to be inserted: ";
      cin>>insert;
      temp_rob_struct = rob_struct(insert);
      //Insertion example
      if(my_rob.push_tail(context, temp_rob_struct) != my_rob.invalid())   
        cout<<"Insertion successful!\n";
      else
        cout<<"Insertion FAILED!\n";
      break;
    case 2:
      cout<<"\nEnter context: ";
      cin>>context;
      //Deletion Example: delete_tail
      if(my_rob.pop_head(context, &ptr_struct))
        cout<<"Deletion success, Value deleted = "<<ptr_struct->opcode<<endl;
      else
        cout<<"Deletion FAILURE\n";
      break;
    case 3:
      cout<<"\nEnter context: ";
      cin>>context;
      //Deletion example: delete_tail
      if(my_rob.pop_tail(context, &ptr_struct))
        cout<<"Deletion success, Value deleted = "<<ptr_struct->opcode<<endl;
      else
        cout<<"Deletion FAILURE\n";
      break;
    case 4:
      for(int c=0; c<num_contexts; c++) {
        cout<<"\nROB for context "<<c<<" :\n";
        cout<<"Head-> ";
        //Head-to-tail traversal example
        for(int index=my_rob.head(c); index!=my_rob.invalid(); index=my_rob.get_next(index)) {
          //Element access example
          ptr_struct = my_rob.get_element(index);
          cout<<"Value="<<ptr_struct->opcode<<" (Slot"<<index<<")    ";
        }
        cout<<"Tail. Occup slots: "<<my_rob.num_entries(c)<<"Slots left: "<<my_rob.num_free_slots_context(c)<<"\n";
      }
      cout<<"Total free slots = "<<my_rob.num_free_slots()<<"\n";
      break;
    case 5:
      cout<<"\nEnter context: ";
      cin>>context;
      cout<<"Head-> ";
      for(int index=my_rob.head(context); index!=my_rob.invalid(); index=my_rob.get_next(index)) {
        ptr_struct = my_rob.get_element(index);
        cout<<"Value="<<ptr_struct->opcode<<" (Slot"<<index<<")    ";
      }
      cout<<"Tail\n";
      break;
    case 6:
      cout<<"Printing contexts in reverse order:\n";
      for(int c=0; c<num_contexts; c++) {
        cout<<"\nROB for context "<<c<<" :\n";
        cout<<"Tail-> ";
        //Tail->Head traversal example
        for(int index=my_rob.tail(c); index!=my_rob.invalid(); index=my_rob.get_prev(index)) {
          ptr_struct = my_rob.get_element(index);
          cout<<"Value = "<<ptr_struct->opcode<<"  ";
        }
        cout<<"Head\n";
      }
      break;
    case 7:
      cout<<"Enter slot to show: ";
      cin>>temp_slot;
      ptr_struct = my_rob.get_element(temp_slot);
      cout<<"Value at slot "<<temp_slot<<" is "<<ptr_struct->opcode<<endl;
      break;
    case 8:
      cout<<"Enter slot to modify: ";
      cin>>temp_slot;
      cout<<"Enter new value: ";
      cin>>insert;
      //Element modification example
      ptr_struct = my_rob.get_element(temp_slot);
      ptr_struct->opcode = insert;
      break;
    case 9:
      cout<<"\nEnter context: ";
      cin>>context;
      my_rob.clear_context(context);
      break;
    case 10:
      my_rob.clear();
      break;
    case 11:
      quit = true;
      break;
    }
  }
  return 1;
}
*/

