#ifndef POLYFLOW_STATS_H
#define POLYFLOW_STATS_H

#include <ostream>
#include <algorithm>
#include <map>
#include <list>
#include <set>

extern uint32_t debug_mask;

class polyflow_stats_c {
private:
  typedef std::map<uint32_t, uint32_t> pc_map_t;
  typedef std::list<uint64_t> ts_list_t;
  typedef std::map<uint32_t, ts_list_t> pc_maplist_t;
  typedef std::map<uint32_t, set<uint32_t> > pc_spawn_map_t;

  pc_map_t spawn_trigger_count;
  pc_map_t spawn_trigger_ignored_count;
  pc_map_t spawn_success_count;
  pc_map_t spawner_success_count;
  pc_map_t spawn_failure_count;
  pc_map_t spawner_failure_count;
  pc_maplist_t spawn_success_ts;
  pc_maplist_t spawn_failure_ts;
  pc_spawn_map_t pc_spawn_map;
  pc_spawn_map_t pc_failed_map;

public:
  polyflow_stats_c() :
    spawn_trigger_count()
  { }

  inline void spawn_trigger(const uint32_t pc) {
    ++spawn_trigger_count[pc];
  }

  inline void spawn_trigger_ignored(const uint32_t pc) {
    ++spawn_trigger_ignored_count[pc];
  }

  inline void spawn_success(const uint32_t pc, const uint64_t ts) {
    ++spawn_success_count[pc];
    spawn_success_ts[pc].push_back(ts);
  }

  inline void spawn_success(const uint32_t spawner_pc, const uint32_t spawned_pc, const uint64_t ts) {
    ++spawn_success_count[spawned_pc];
    ++spawner_success_count[spawner_pc];
    spawn_success_ts[spawned_pc].push_back(ts);
    bool in_spawned_set = 
        (pc_spawn_map[spawner_pc].find(spawned_pc) != pc_spawn_map[spawner_pc].end());
    if (!in_spawned_set)
        pc_spawn_map[spawner_pc].insert(spawned_pc);
  }

  inline void spawn_failure(const uint32_t spawner_pc, const uint32_t spawned_pc, const uint64_t ts) {
    ++spawn_failure_count[spawned_pc];
    ++spawner_failure_count[spawner_pc];
    spawn_failure_ts[spawned_pc].push_back(ts);
    bool in_spawned_set = 
        (pc_failed_map[spawner_pc].find(spawned_pc) != pc_failed_map[spawner_pc].end());
    if (!in_spawned_set)
        pc_failed_map[spawner_pc].insert(spawned_pc);
  }

  void print() const {
    print(cout);
  }

  void print_spawn_success_info(ostream& p) const
  {
      p << "Successful (Spawner PC, Spawned PC) sets : \n";
      pc_spawn_map_t::const_iterator pcitr;
      for (pcitr = pc_spawn_map.begin(); pcitr != pc_spawn_map.end(); ++pcitr) {
          uint32_t pc = pcitr->first;
          uint32_t count = spawner_success_count.find(pc)->second;
          p << "\tcount= " << dec << count << " PC = " << hex << showbase << pcitr->first << " : ";
          const set<uint32_t> &spawned = pcitr->second;
          //copy(pcitr->second.begin(), pcitr->second.end(), ostream_iterator<const char*>(p, " "));
          copy(spawned.begin(), spawned.end(), ostream_iterator<uint32_t>(p, " "));
          p << endl;
      }
  }

  void print_spawn_failure_info(ostream& p) const
  {
      p << "UnSuccessful (Spawner PC, Spawned PC) sets : \n";
      pc_spawn_map_t::const_iterator pcitr;
      for (pcitr = pc_failed_map.begin(); pcitr != pc_failed_map.end(); ++pcitr) {
          uint32_t pc = pcitr->first;
          uint32_t count = spawner_failure_count.find(pc)->second;
          p << dec << "\tcount= " << count << " PC = " << hex << showbase << pcitr->first << " : ";
          const set<uint32_t> &spawned = pcitr->second;
          //copy(pcitr->second.begin(), pcitr->second.end(), ostream_iterator<const char*>(p, " "));
          copy(spawned.begin(), spawned.end(), ostream_iterator<uint32_t>(p, " "));
          p << endl;
      }
    p << "Spawn Failure Counts:\n";
    for(pc_map_t::const_iterator pcitr = spawn_failure_count.begin(); pcitr != spawn_failure_count.end(); ++pcitr) {
      p << "\tSpawned PC: " << hex << showbase << pcitr->first 
        << " Count: " << dec << pcitr->second << endl;
    }
  }

  void print(ostream& p) const {
    print_spawn_success_info(p);
    print_spawn_failure_info(p);

    p << "Spawn Success Counts:\n";
    for(pc_map_t::const_iterator pcitr = spawn_success_count.begin(); pcitr != spawn_success_count.end(); ++pcitr) {
      p << "\tPC: " << hex << showbase << pcitr->first 
        << " Count: " << dec << pcitr->second << endl;
    }

    if(debug_mask) {
      p << "Spawn Success Timestamps:\n";
      for(pc_maplist_t::const_iterator pcitr = spawn_success_ts.begin(); pcitr != spawn_success_ts.end(); ++pcitr) {
        p << "PC: " << hex << showbase << pcitr->first;
        for(ts_list_t::const_iterator litr = pcitr->second.begin(); litr != pcitr->second.end(); ++litr) {
          p << " ts: " << dec << *litr;
        }
        p << endl;
      }
    }

    uint32_t sum = 0;
    p << "Spawn Trigger Counts:\n";
    for(pc_map_t::const_iterator pcitr = spawn_trigger_count.begin(); pcitr != spawn_trigger_count.end(); ++pcitr) {
      p << "PC: " << hex << showbase << pcitr->first 
        << " Count: " << dec << pcitr->second << endl;
      sum += pcitr->second;
    }
    p << "total: " << sum << endl;
    
    sum = 0;
    p << "Spawn Trigger Ignored Counts:\n";
    for(pc_map_t::const_iterator pcitr = spawn_trigger_ignored_count.begin(); pcitr != spawn_trigger_ignored_count.end(); ++pcitr) {
      p << "PC: " << hex << showbase << pcitr->first 
        << " Count: " << dec << pcitr->second << endl;
      sum += pcitr->second;
    }
    p << "total: " << sum << endl;
  }
};

#endif //POLYFLOW_STATS_H
