#ifndef ISSUE_BUFFER_H
#define ISSUE_BUFFER_H

#include <inttypes.h>
#include "decode.h"
#include "free_list.h"

extern uint32_t debug_mask;

template<typename T>
class issue_buffer_slot_c {
public:
  T entry;
  size_t prev;
  size_t next;

  issue_buffer_slot_c() : 
    entry(), prev(0), next(0) { }
};

template<typename T>
class issue_buffer_c {
public:
  issue_buffer_c(size_t);
  size_t insert(const T&);
  void remove(size_t slot);
  T& get(size_t);
  size_t get_next(size_t) const;  
  void clear();
  void print() const;
  bool empty() const;
  size_t num_free() const;
  size_t num_used() const;
  size_t get_head() const;
  size_t get_tail() const;
  size_t get_end() const;

private:
  vector<issue_buffer_slot_c<T> > list;
  size_t  size;
  size_t  head;
  size_t  tail;
  free_list free_slots;
};

template<typename T>
issue_buffer_c<T>::issue_buffer_c(size_t x_size) : 
  list(x_size+1), 
  size(x_size), 
  head(x_size), 
  tail(x_size), 
  free_slots(x_size, 0)
{ }

template <typename T>
size_t issue_buffer_c<T>::insert(const T& x_entry) {
  // get the tag of a free slot in the scoreboard
  size_t free_slot = 0;
  free_slots.get_free_obj(free_slot);

  if (empty()) {
    head = free_slot;
    tail = free_slot;
    
    list[free_slot].prev  = size;
    list[free_slot].next  = size;
  }
  else {
    // set the tail slot's next ptr to point to the free slot
    list[tail].next = free_slot;

    // set the free slot's prev ptr to point to the tail,
    list[free_slot].prev  = tail;

    // set the tail to point to the new instr's slot
    tail            = free_slot;
    list[tail].next = size;
  }

  list[free_slot].entry = x_entry;
  //  list[free_slot].instr->sb_slot = free_slot;

  return free_slot;
}

template<typename T>
void issue_buffer_c<T>::remove(size_t x_slot) {
  if(debug_mask & 0x0008) printf("issue_buffer_c::remove slot %u\n", x_slot);
  // remove the slot from the list by resetting the pointers
  // of the next slot and the prev slot to point to each other

  //  assert(list[x_slot].entry != T());
  //  list[x_slot].entry = T();

  if (x_slot == head) {
    head = list[x_slot].next;
    list[head].prev = size;
  }
  else {
    list[list[x_slot].prev].next = list[x_slot].next;
  }

  if (x_slot == tail) {
    tail = list[x_slot].prev;
    list[tail].next = size;
  }
  else {
    list[list[x_slot].next].prev = list[x_slot].prev;
  }

  free_slots.put_free_obj(x_slot);
}

template<typename T>
T& issue_buffer_c<T>::get(size_t x_slot) {
  return list[x_slot].entry;
}

template<typename T>
size_t issue_buffer_c<T>::get_next(size_t x_slot) const {
  return list[x_slot].next;
}

template<typename T>
void issue_buffer_c<T>::clear() {
  for (size_t iter = head; iter != size; iter = list[iter].next) {
    free_slots.put_free_obj(iter);
  }
  head = size;
  tail = size;
  list[head].prev = size;
  list[head].next = size;
}

template<typename T>
void issue_buffer_c<T>::print() const {
  printf("scoreboard list: size %u head %u tail %u\n", size, head, tail);
  for (size_t slot = 0; slot < size+1; slot++) {
    printf("slot %u : prev %u next %u\n", slot, list[slot].prev, list[slot].next);
  }
}

template<typename T>
bool issue_buffer_c<T>::empty() const {
  return (head == size);
}

template<typename T>
size_t issue_buffer_c<T>::num_free() const {
  return free_slots.num_free_objs();
}

template<typename T>
size_t issue_buffer_c<T>::num_used() const {
  return size - free_slots.num_free_objs();
}

template<typename T>
size_t issue_buffer_c<T>::get_head() const {
  return head;
}

template<typename T>
size_t issue_buffer_c<T>::get_tail() const {
  return tail;
}

template<typename T>
size_t issue_buffer_c<T>::get_end() const {
  return size;
}

#endif
