#include "inorder_scheduler.h"

//Debug mask for this stage
uint32_t io_schd_mask = 0x00800000;

uint32_t inorder_scheduler::pre_fifo_size = 10;
const uint32_t inorder_scheduler::pre_fifo_unused_limit = 2;

inline void inorder_scheduler::handle_writeback() 
{
  if (writeback_bus().reorder_slot != INV_POOL_PTR) {
    //update scoreboard for arch_dest_reg
    score_board[writeback_bus().arch_tag] = true;
  }
  /*
  //All these are not needed anymore since we dont have a issue_buffer
  if (branch_exec().is_branch) {
    // jals and jalrs get completed by writeback_bus, not branch_exec
    if (branch_exec().is_call) {
      if(debug_mask & 0x0008) printf("ioSCHD> branch bus: ");
    }
  }
  // mark stores
  if (store_bus().reorder_slot != INV_POOL_PTR) {
    if(debug_mask & 0x0008) printf("ioSCHD> store bus: ");
  }
  // mark syscalls
  if (syscall_exec().valid) {
    //assert(syscall_exec().instr_num == held_instr->instr_num);
    if(debug_mask & 0x0008) printf("ioSCHD> syscall bus: ");
  }
  */
}

inline void inorder_scheduler::handle_misspeculation() 
{
  if (simpanic_in().valid || branch_exec().mispredict || 
      mem_access_exec().misspeculation || syscall_exec().valid) {
    held_instr = decoder::noop;
    schd_state = EMPTY;    
    if(simpanic_in().valid) {
      //recreate the score_board
      for(uint32_t r = 0; r < NUM_ARCH_REGS; r++)
        score_board[r] = true;
    }
    else if (mem_access_exec().misspeculation) {
      //sts and lds are issued in order because of which
      //stlf doesnt cause mis-speculations
    }
    else if(branch_exec().mispredict) {
      //undo the changes made to the scoreboard becuz of the branch
      decoder::instr_h instr = instr_out();
      if (!instr->noop) {
        assert(instr->instr_num == (branch_exec().instr_num + 1));
        //printf("io_schd: score_board may need fixing for branch_exec\n");
        if (instr->arch_dest_reg != 0) {
          //printf("io_schd: fixing score_board\n");
          //instr->print("io_schd: Culprit");
          //printf("\n");
          assert((score_board[instr->arch_dest_reg] == false));
          score_board[instr->arch_dest_reg] = true;
        }
      }
    }
    else if (syscall_exec().valid) {
      //undo the changes made to the scoreboard becos of the syscall
      for(uint32_t r = 0; r < NUM_ARCH_REGS; r++)
        score_board[r] = true;
    }
  }
}

bool inorder_scheduler::syscall_check(decoder::instr_h instr)
{
  //syscalls are only issue when they reach the rob head
  if (instr->is_syscall)
    return (rob_head()[instr->context] == instr->reorder_slot);
  else
    return true;
}

void inorder_scheduler::handle_issue_dispatch()
{
  if (simpanic_in().valid || branch_exec().mispredict || 
        mem_access_exec().misspeculation || syscall_exec().valid) 
  {
    //On mis-speculation ignore held instr send noop to exec
    instr_out = decoder::noop;
    //setting correct state, flushing etc. is performed by handle_misspeculation
  }
  else if(schd_state == WAITING) {
    //Check if held instr ready ?
    bool ready = score_board[held_instr->arch_s_reg] && 
      score_board[held_instr->arch_t_reg] && 
      score_board[held_instr->arch_dest_reg] &&
      syscall_check(held_instr);
    if (ready) {
      //ISSUE
      instr_out = held_instr;
      score_board[held_instr->arch_dest_reg] = false;
      held_instr = decoder::noop;
      schd_state = EMPTY;
    } else {
      //SEND NOOP
      instr_out = decoder::noop;
    }
  }
  else if(schd_state == EMPTY) {
    decoder::instr_h instr;
    //instr = instr_in();
    //if (!instr_in()->noop) 
    if (pre_fifo.size() > 0) 
      instr = pre_fifo.front();
    else instr = decoder::noop;
    if (!instr->noop)
    {
      pre_fifo.pop_front(); //remove instr from fifo
      held_instr = instr; 
      schd_state = WAITING;
      //Check if ready
      bool ready = score_board[held_instr->arch_s_reg] && 
        score_board[held_instr->arch_t_reg] && 
        score_board[held_instr->arch_dest_reg] &&
        syscall_check(held_instr);
      if (ready) 
      {
        //ISSUE
        instr_out = held_instr;
        score_board[held_instr->arch_dest_reg] = false;
        held_instr = decoder::noop;
        schd_state = EMPTY;
      } else {
        //SEND NOOP
        instr_out = decoder::noop;
      }
    } else {
      //SEND NOOP
      instr_out = decoder::noop;
    }
  } else {
    //illegal schd state
  }
}

void inorder_scheduler::set_stall_conditions() 
{
  bool lsq_stall = (the_exec_unit->the_stq.stall_stores() || 
      the_exec_unit->the_ldq.stall_loads());
  //bool dependency_stall = (schd_state == WAITING);
  bool dependency_stall = false;;
  //FIXME: do we still need the dependency_stall ???
  bool pre_fifo_stall = 
    (pre_fifo_size - pre_fifo.size()) <= pre_fifo_unused_limit;
  scoreboard_stall = dependency_stall || lsq_stall || pre_fifo_stall;
}

char* to_str(schd_state_t &st) 
{
  if (st == WAITING) {
    return "waiting";
  } 
  else if (st == EMPTY) {
    return "empty";
  }
  else {
    return "UNKNOWN";
  }
}

void inorder_scheduler::print_debug()
{
  if (debug_mask & io_schd_mask) {
    printf("io_schd: cc(%llu) state(%s) stall(%c)\n",
        cycle_count,to_str(schd_state),('0'+scoreboard_stall()));
    if (!held_instr->noop) {
      held_instr->print("io_schd: held_instr");
      printf("\n");
    }
    if (!instr_in()->noop) {
      instr_in()->print("io_schd: instr_in");
      printf("\n");
    }
    if (!instr_out()->noop) {
      instr_out()->print("io_schd: instr_out");
      printf("\n");
    }
    if (!instr_in()->noop || !held_instr->noop || !instr_out()->noop) {
      for(uint32_t r = 0; r < NUM_ARCH_REGS; r++) {
        printf("(%2u-%1d) ",r,(score_board[r]==true));
        if(((r+1)%16)==0) printf("\n");
      }
    }
    if (branch_exec().mispredict) {
      branch_exec().print();
    }
    if (writeback_bus().reorder_slot != INV_POOL_PTR) {
      writeback_bus().print("io_schd:");
    }
    if (pre_fifo.size() > 0) {
      deque<decoder::instr_h>::const_iterator i = pre_fifo.begin();
      printf("io_schd: pre_fifo %d\n",pre_fifo.size());
      for (int j=0; i != pre_fifo.end(); i++,j++) {
        printf("%2d: ",j);
        (*i)->print("");
        printf("\n");
      }
    }
    printf("io_schd:_ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _\n");
  }
}

void inorder_scheduler::handle_pre_fifo()
{
  //Assign LSQ entries to all eligible instructions even if cancelled
  if (!instr_in()->noop) {
    decoder::instr_h instr = instr_in();
    //allocate lsq entries
    if (instr->is_store) {
      instr->stq_slot = 
        the_exec_unit->the_stq.allocate(instr->context, instr->instr_num);
    }
    else if (instr->is_load) {
      instr->ldq_slot = 
        the_exec_unit->the_ldq.allocate(instr->context, instr->instr_num);
    }
  }
  //Add new instrs to scheduler pre_fifo
  if (simpanic_in().valid || branch_exec().mispredict || 
      mem_access_exec().misspeculation || syscall_exec().valid) 
  {
    //ignore input instr and flush the fifo
    pre_fifo.clear();
  } else {
    if (!instr_in()->noop) {
      pre_fifo.push_back(instr_in());
    }
  }
} 

void inorder_scheduler::recalc() 
{
  score_board[0] = true;
  print_debug();

  //1.
  //handle various writebacks
  //update the scoreboard
  handle_writeback();

  //2.
  //handle misspeculation
  //fix the scoreboard
  //flush held instr
  handle_misspeculation();

  //2.1
  //maintain fifo
  //fill in the fifo from instr_in
  //flush fifo on mis-speculations
  handle_pre_fifo();
  
  //3.
  //handle dispatch and issue
  //if state empty - take the instr from fifo and test it for issue
  //if state waiting - take held instr and test it for issue
  handle_issue_dispatch();

  //6.
  //set stall signals
  //stall the front end if we couldnt issue the instruction being held.
  //
  //Note: Ideally the instr on the input latches should remain there untouched
  //but the rename_stage seems to be writing noop's on backend stalls
  //because of which the fifo needs to be filled even during stalls
  set_stall_conditions();
}

