#include "frontend_delay_stage.h"

void frontend_delay_stage::recalc() {
    
  //we always write an instr_in, but only update the head when there is no stall
  //put the instr in now so that our misspeculation check and context kill check can kill
  //something as it is written in
  instrq[qhead] = instr_in();

  if(instr_out()->noop || 
     simpanic_in().mispredicted_context(instr_out()->context) ||
     branch_exec().mispredicted_context(instr_out()->context) ||
     mem_access_exec().mispredicted_context(instr_out()->context)||
     (syscall_exec().valid && syscall_exec().context == instr_out()->context))
    {
      // state: stall detected and misprediction or misspeculation detected in the thread that
      //  owns the instruction that is in the output latch
      // action: flush the instruction in the output latch

      decoder::instr_h temp_instr = decoder::noop;
      instr_out = temp_instr;
    }
    
  //check if we need to kill any contexts or if there was some kind of misspeculation
  if (simpanic_in().valid || branch_exec().mispredict || mem_access_exec().misspeculation || syscall_exec().valid) {
      
    //fprintf(stderr, "got kill signal in frontend_delay in cycle %lld\n", cycle_count);

    //the different signals above are NOT mutually exclusive.
      
    uint32_t cancel_context  = 0;
    uint64_t cancel_ts       = 0ULL;

    //even if more than one of the flush signals is valid, its okay (i.e, no need to enforce ordering on which signal should be handled first
    if(simpanic_in().valid) {
      cancel_context = simpanic_in().context;
      cancel_ts = simpanic_in().instr_num + 1;  //cancel all instructions AFTER the simpanic
      flush_context(cancel_context, cancel_ts);
    }
    if (mem_access_exec().misspeculation) {
      cancel_context = mem_access_exec().recovery_context;
      cancel_ts      = mem_access_exec().recovery_ts;
      flush_context(cancel_context, cancel_ts);
    }
    if(branch_exec().mispredict) {
      cancel_context  = branch_exec().context;
      cancel_ts       = branch_exec().instr_num + 1;
      flush_context(cancel_context, cancel_ts);
    }
    if(syscall_exec().valid) {
      cancel_context = syscall_exec().context;
      cancel_ts      = syscall_exec().instr_num + 1;
      flush_context(cancel_context, cancel_ts);
    }
      
    // flush the instructions that belong to the mispredicting/misspeculating context

      
  }
    
  if (!scoreboard_stall() && !rename_stall() && !rob_stall()[0]) {
    instr_out = instrq[(qhead+1)%qsize];
    qhead = (qhead + 1) % qsize;
  }

  //update the output statereg
  instr_counts = inst_count;

  if (debug_mask & 0x10000)
    print_queue();

}

void frontend_delay_stage::print_queue() {

  fprintf(stdout, "qhead: %d\n", qhead);

  fprintf(stdout, "inst_counts:\n");
  for (uint32_t i=0; i<num_contexts; i++)
    fprintf(stdout, "context[%d]: %d\n", i, inst_count[i]);

  uint32_t i=(qhead+1)%qsize;
  do {

    fprintf(stdout, "slot %d: pc %x noop %d context %d\n", i, instrq[i]->program_counter, instrq[i]->noop, instrq[i]->context);

    i = (i+1)%qsize;
  } while (i != qhead);

    

  fprintf(stdout, "---------------------------\n");

}

void frontend_delay_stage::reset_totals() {


  for (uint32_t i=0; i<num_contexts; i++)
    inst_count[i] = 0;

  //add up the totals for each context, the qhead location is not actually in the queue
  for (uint32_t i=0; i<qsize; i++) {
    if ((instrq[i]->context < num_contexts) && (i != qhead))
      inst_count[instrq[i]->context]++;
  }

  if (debug_mask & 0x10000) {
    fprintf(stdout, "queue after kill\n");
    print_queue();
  }

  if (debug_mask & 0x10000)
    fprintf(stdout, "totals reset\n");

}


//flush all instructions from a context
void frontend_delay_stage::flush_context(uint32_t x_cancel_context, uint64_t x_cancel_ts) {
  // flush the instructions that belong to the mispredicting/misspeculating context
  for(uint32_t i=0; i<qsize; i++) {
    if ((instrq[i]->context == x_cancel_context) && (instrq[i]->instr_num >= x_cancel_ts)) {
      if (debug_mask & 0x10000)
        fprintf(stderr, "killing inst %lld\n", instrq[i]->instr_num);
      decoder::instr_h l_instr = instrq[i];
      l_instr = decoder::noop;
      instrq[i] = l_instr;
    }
  }
}
