#include "execute.h"
#include <string.h>

// Global vars for debugging
extern uint32_t debug_mask;

static char err_msg_buf[1024];

void
datapath::execute(decoder::instr_h instr)
{
  uint32_t next_program_counter = state->program_counter() + 4;

  switch (instr->opcode)
  {
  case decoder::sll: {
    state->dest(instr) = sign_extend64(t_op32(instr) << instr->shamt_immediate());
    break;
  }
  case decoder::srl: {
    state->dest(instr) = sign_extend64(t_op32(instr) >> instr->shamt_immediate());
    break;
  }
  case decoder::sra: {
    state->dest(instr) = sign_extend64((int32_t)(((int32_t)(t_op32(instr))) >> instr->shamt_immediate()));
    break;
  }
  case decoder::sllv: {
    // note that sllv switches the usual order of rs and rt
    state->dest(instr) = sign_extend64(t_op32(instr) << (s_op32(instr) & 0x1F));
    break;
  }
  case decoder::srlv: {
    // note that srlv switches the usual order of rs and rt
    state->dest(instr) = sign_extend64(t_op32(instr) >> (s_op32(instr) & 0x1F));
    break;
  }
  case decoder::srav: {
    // note that srav switches the usual order of rs and rt
    state->dest(instr) = sign_extend64((int32_t)(((int32_t)(t_op32(instr))) >> (s_op32(instr) & 0x1F)));
    break;
  }
  case decoder::dsll: {
    state->dest(instr) = state->t_op64(instr) << instr->shamt_immediate();
    break;
  }
  case decoder::dsrl: {
    state->dest(instr) = state->t_op64(instr) >> instr->shamt_immediate();
    break;
  }
  case decoder::dsra: {
    state->dest(instr) = (int64_t)(((int64_t)(state->t_op64(instr))) >> instr->shamt_immediate());
    break;
  }
  case decoder::dsllv: {
    // note that sllv switches the usual order of rs and rt
    state->dest(instr) = state->t_op64(instr) << (s_op32(instr) & 0x3F);
    break;
  }
  case decoder::dsrlv: {
    // note that srlv switches the usual order of rs and rt
    state->dest(instr) = state->t_op64(instr) >> (s_op32(instr) & 0x3F);
    break;
  }
  case decoder::dsrav: {
    // note that srav switches the usual order of rs and rt
    state->dest(instr) = (int64_t)(((int64_t)(state->t_op64(instr))) >> (s_op32(instr) & 0x3F));
    break;
  }
  case decoder::jr: {
    next_program_counter = s_op32(instr);
    break;
  }
  case decoder::jalr: {
    state->dest(instr) = sign_extend64(state->program_counter() + 4);
    next_program_counter = s_op32(instr);
    break;
  }
  case decoder::mull: {
    state->dest(instr) = sign_extend64(s_op32(instr) * t_op32(instr));
    break;
  }
  case decoder::div: {
    int32_t dividend = 0;

    if (t_op32(instr) == 0) {
      state->print_err_msg("arch_model: divide by zero\n");
    }
    else {
      dividend = (int32_t)(s_op32(instr)) / (int32_t)(t_op32(instr));
    }
    state->dest(instr) = sign_extend64(dividend);
    break;
  }
  case decoder::divu: {
    uint32_t dividend = 0;

    if (t_op32(instr) == 0) {
      state->print_err_msg("arch_model: divide by zero\n");
    }
    else {
      dividend = s_op32(instr) / t_op32(instr);
    }
    state->dest(instr) = sign_extend64(dividend);
    break;
  }
  case decoder::mulh: {
    int64_t rs_l = (int32_t)(s_op32(instr));
    int64_t rt_l = (int32_t)(t_op32(instr));
    int64_t result = rs_l * rt_l;
    state->dest(instr) = sign_extend64((uint32_t)(result >> 32));
    break;
  }
  case decoder::mulhu: {
    uint64_t rs_l = s_op32(instr);
    uint64_t rt_l = t_op32(instr);
    uint64_t result = rs_l * rt_l;
    state->dest(instr) = sign_extend64((uint32_t)(result >> 32));
    break;
  }
  case decoder::dmull: {
    state->dest(instr) = state->s_op64(instr) * state->t_op64(instr);
    break;
  }
  case decoder::ddiv: {
    int64_t dividend = 0;

    if (state->t_op64(instr) == 0) {
      state->print_err_msg("arch_model: divide by zero\n");
    }
    else {
      dividend = (int64_t)(state->s_op64(instr)) / (int64_t)(state->t_op64(instr));
    }
    state->dest(instr) = dividend;
    break;
  }
  case decoder::ddivu: {
    uint64_t dividend = 0;

    if (state->t_op64(instr) == 0) {
      state->print_err_msg("arch_model: divide by zero\n");
    }
    else {
      dividend = state->s_op64(instr) / state->t_op64(instr);
    }
    state->dest(instr) = dividend;
    break;
  }
  case decoder::dmulh: {
    state->print_err_msg("arch_model: dmulh not implemented yet!\n");
    break;
  }
  case decoder::dmulhu: {
    state->print_err_msg("arch_model: dmulhu not implemented yet!\n");
    break;
  }
  case decoder::addu: {
    state->dest(instr) = sign_extend64(s_op32(instr) + t_op32(instr));
    break;
  }
  case decoder::subu: {
    state->dest(instr) = sign_extend64(s_op32(instr) - t_op32(instr));
    break;
  }
  case decoder::uand: {
    state->dest(instr) = state->s_op64(instr) & state->t_op64(instr);
    break;
  }
  case decoder::uor: {
    state->dest(instr) = state->s_op64(instr) | state->t_op64(instr);
    break;
  }
  case decoder::uxor: {
    state->dest(instr) = state->s_op64(instr) ^ state->t_op64(instr);
    break;
  }
  case decoder::nor: {
    state->dest(instr) = ~(state->s_op64(instr) | state->t_op64(instr));
    break;
  }
  case decoder::daddu: {
    state->dest(instr) = state->s_op64(instr) + state->t_op64(instr);
    break;
  }
  case decoder::dsubu: {
    state->dest(instr) = state->s_op64(instr) - state->t_op64(instr);
    break;
  }
  case decoder::slt: {
    state->dest(instr) = (((int64_t)(state->s_op64(instr))) < ((int64_t)(state->t_op64(instr))));
    break;
  }
  case decoder::sltu: {
    state->dest(instr) = state->s_op64(instr) < state->t_op64(instr);
    break;
  }
  case decoder::beq: {
    if (state->s_op64(instr) == state->t_op64(instr)) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::bne: {
    if (state->s_op64(instr) != state->t_op64(instr)) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::addiu: {
    state->dest(instr) = sign_extend64(s_op32(instr) + instr->signed_immediate32());
    break;
  }
  case decoder::slti: {
    state->dest(instr) = (((int64_t)(state->s_op64(instr))) < instr->signed_immediate64());
    break;
  }
  case decoder::sltiu: {
    state->dest(instr) = state->s_op64(instr) < ((uint64_t)(instr->signed_immediate64()));
    break;
  }
  case decoder::andi: {
    state->dest(instr) = state->s_op64(instr) & instr->unsigned_immediate();
    break;
  }
  case decoder::ori: {
    state->dest(instr) = state->s_op64(instr) | instr->unsigned_immediate();
    break;
  }
  case decoder::xori: {
    state->dest(instr) = state->s_op64(instr) ^ instr->unsigned_immediate();
    break;
  }
  case decoder::aui: {
    state->dest(instr) = sign_extend64(s_op32(instr) + (instr->unsigned_immediate() << 16));
    break;
  }
  case decoder::daddiu: {
    state->dest(instr) = state->s_op64(instr) + instr->signed_immediate64();
    break;
  }
  case decoder::lb: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_int8(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lb: %x %llx\n", vaddr, result);
      printf("lb: %8x %2llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::lh: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_int16(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lh: %x %llx\n", vaddr, result);
      printf("lh: %8x %4llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::lw: {
    uint32_t vaddr =s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_int32(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lw: %x %llx\n", vaddr, result);
      printf("lw: %8x %8llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::ld: {
    uint32_t vaddr =s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_uint64(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "ld  %x %llx\n", vaddr, result);
      printf("ld: %8x %16llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::lbu: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_uint8(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lbu: %x %llx\n", vaddr, result);
      printf("lbu: %8x %2llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::lhu: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_uint16(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lhu: %x %llx\n", vaddr, result);
      printf("lhu: %8x %4llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::lwu: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->is_load = true;
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    uint64_t result = state->load_uint32(vaddr, instr->context);
    state->dest(instr) = result;
    if (debug_mask & 0x1000) {
      sprintf(err_msg_buf, "lwu: %x %llx\n", vaddr, result);
      printf("lwu: %8x %8llx\n", vaddr, result);
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::sb: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    state->store_uint8((uint8_t)(state->t_op64(instr)), vaddr, instr->context);
    if (debug_mask & 0x0800) {
      sprintf(err_msg_buf, "sb: %x %x\n", vaddr, t_op32(instr));
      printf("sb: %8x %2x\n", vaddr, t_op32(instr));
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::sh: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    state->store_uint16((uint16_t)(state->t_op64(instr)), vaddr, instr->context);
    if (debug_mask & 0x0800) {
      sprintf(err_msg_buf, "sh: %x %x\n", vaddr, t_op32(instr));
      printf("sh: %8x %4x\n", vaddr, t_op32(instr));
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::sw: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    state->store_uint32((uint32_t)(state->t_op64(instr)), vaddr, instr->context);
    if (debug_mask & 0x0800) {
      sprintf(err_msg_buf, "sw: %x %x\n", vaddr, t_op32(instr));
      printf("sw: %8x %8x\n", vaddr, t_op32(instr));
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::sd: {
    uint32_t vaddr = s_op32(instr) + instr->signed_immediate32();
    instr->vaddr = vaddr;
    instr->base_reg_val = s_op32(instr);
    state->store_uint64(state->t_op64(instr), vaddr, instr->context);
    if (debug_mask & 0x0800) {
      sprintf(err_msg_buf, "sd: %x %llx\n", vaddr, state->t_op64(instr));
      printf("sd: %8x %16llx\n", vaddr, state->t_op64(instr));
      state->print_err_msg(err_msg_buf);
    }
    break;
  }
  case decoder::bltz: {
    if (((int64_t)(state->s_op64(instr))) < 0) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::blez: {
    if (((int64_t)(state->s_op64(instr))) <= 0) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::bgez: {
    if (((int64_t)(state->s_op64(instr))) >= 0) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::bgtz: {
    if (((int64_t)(state->s_op64(instr))) > 0) {
      next_program_counter = state->program_counter() + instr->branch_immediate();
    }
    else next_program_counter = state->program_counter() + 4;
    break;
  }
  case decoder::j: {
    next_program_counter = instr->jump_immediate();
    break;
  }
  case decoder::jal: {
    state->dest(instr) = sign_extend64(state->program_counter() + 4);
    next_program_counter = instr->jump_immediate();
    break;
  }
  case decoder::add_s: {
    state->dest(instr) = bitops::cvt_float2int(bitops::cvt_int2float(s_op32(instr)) + bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::sub_s: {
    state->dest(instr) = bitops::cvt_float2int(bitops::cvt_int2float(s_op32(instr)) - bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::mul_s: {
    state->dest(instr) = bitops::cvt_float2int(bitops::cvt_int2float(s_op32(instr)) * bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::div_s: {
    state->dest(instr) = bitops::cvt_float2int(bitops::cvt_int2float(s_op32(instr)) / bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::abs_s: {
    float rs_f = bitops::cvt_int2float(s_op32(instr));
    state->dest(instr) = bitops::cvt_float2int((rs_f < 0.0f) ? -rs_f : rs_f);
    break;
  }
  case decoder::neg_s: {
    state->dest(instr) = bitops::cvt_float2int(-bitops::cvt_int2float(s_op32(instr)));
    break;
  }
  case decoder::trunc_l_s: {
    state->dest(instr) = ((int64_t)(bitops::cvt_int2float(s_op32(instr))));
    break;
  }
  case decoder::trunc_w_s: {
    state->dest(instr) = sign_extend64(((int32_t)(bitops::cvt_int2float(s_op32(instr)))));
    break;
  }
  case decoder::cvt_s_l: {
    state->dest(instr) = bitops::cvt_float2int((float)((int64_t)(state->s_op64(instr))));
    break;
  }
  case decoder::cvt_l_s: {
    state->dest(instr) = ((int64_t)(bitops::cvt_int2float(s_op32(instr))));
    break;
  }
  case decoder::c_eq_s: {
    state->dest(instr) = (bitops::cvt_int2float(s_op32(instr)) == bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::c_lt_s: {

    state->dest(instr) = (bitops::cvt_int2float(s_op32(instr)) < bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::c_le_s: {
    state->dest(instr) = (bitops::cvt_int2float(s_op32(instr)) <= bitops::cvt_int2float(t_op32(instr)));
    break;
  }
  case decoder::add_d: {
    state->dest(instr) = bitops::cvt_double2int(bitops::cvt_int2double(state->s_op64(instr)) + bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::sub_d: {
    state->dest(instr) = bitops::cvt_double2int(bitops::cvt_int2double(state->s_op64(instr)) - bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::mul_d: {
    state->dest(instr) = bitops::cvt_double2int(bitops::cvt_int2double(state->s_op64(instr)) * bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::div_d: {
    state->dest(instr) = bitops::cvt_double2int(bitops::cvt_int2double(state->s_op64(instr)) / bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::abs_d: {
    float rs_f = bitops::cvt_int2double(state->s_op64(instr));
    state->dest(instr) = bitops::cvt_double2int((rs_f < 0.0f) ? -rs_f : rs_f);
    break;
  }
  case decoder::neg_d: {
    state->dest(instr) = bitops::cvt_double2int(-bitops::cvt_int2double(state->s_op64(instr)));
    break;
  }
  case decoder::trunc_l_d: {
    state->dest(instr) = ((int64_t)(bitops::cvt_int2double(state->s_op64(instr))));
    break;
  }
  case decoder::trunc_w_d: {
    state->dest(instr) = sign_extend64(((int32_t)(bitops::cvt_int2double(state->s_op64(instr)))));
    break;
  }
  case decoder::cvt_d_l: {
    state->dest(instr) = bitops::cvt_double2int((double)((int64_t)(state->s_op64(instr))));
    break;
  }
  case decoder::cvt_l_d: {
    state->dest(instr) = ((int64_t)(bitops::cvt_int2double(state->s_op64(instr))));
    break;
  }
  case decoder::c_eq_d: {
    state->dest(instr) = (bitops::cvt_int2double(state->s_op64(instr)) == bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::c_lt_d: {
    state->dest(instr) = (bitops::cvt_int2double(state->s_op64(instr)) < bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::c_le_d: {
    state->dest(instr) = (bitops::cvt_int2double(state->s_op64(instr)) <= bitops::cvt_int2double(state->t_op64(instr)));
    break;
  }
  case decoder::cvt_d_s: {
    state->dest(instr) = bitops::cvt_double2int((double)(bitops::cvt_int2float(s_op32(instr))));
    break;
  }
  case decoder::cvt_s_d: {
    state->dest(instr) = bitops::cvt_float2int((float)(bitops::cvt_int2double(state->s_op64(instr))));
    break;
  }
  case decoder::syscall_halt: {
    state->halt();
    state->print_err_msg("processor halted\n");
    break;
  }
  case decoder::syscall_unix: {
    state->emulate_syscall(s_op32(instr), instr->context);
    break;
  }
  case decoder::syscall_ftostr: {
    char format[32];
    char result[1024];
    sprintf(format, "%%.%de", s_op32(instr));
    sprintf(result, format, bitops::cvt_int2double(state->dest(instr)));
    state->strcpy_from_host(t_op32(instr), result, instr->context);
    break;
  }
  case decoder::syscall_strtof: {
    char str[4096];
    state->strcpy_to_host(str, s_op32(instr), instr->context);
    state->dest(instr) = bitops::cvt_double2int((double)atof(str));
    printf("executed strtof, got value %e\n", bitops::cvt_int2double(state->dest(instr)));
    break;
  }
  case decoder::signaling_noop: {
    sprintf(err_msg_buf, "illegal instruction: %08x: %08x\n", state->program_counter(), instr->instr);
    state->print_err_msg(err_msg_buf);
    break;
  }
  default: {
    sprintf(err_msg_buf, "unknown opcode: %08x: opcode %d, instr %08x\n", state->program_counter(), instr->opcode, instr->instr);

    state->print_err_msg(err_msg_buf);
    break;
  }
  } // end switch

  state->zero_reg_0();

  instr->wb_data = state->dest(instr);

  state->program_counter() = next_program_counter;
} // end execute function

#include "elf_loader.h"

// SMT: DO NOT need to pass in context
// memcpy_functor is never instantiated with a pointer
// to a pipelined-model. memcpy_functor is always instantiated
// with a pointer to an arch-model. so the state_semantics associated
// with a memcpy_functor is always an arch-model.
struct memcpy_functor
{
  state_semantics* the_mem;
  memcpy_functor(state_semantics* m) : the_mem(m) { }
  void operator()(uint32_t dest, const void* src, size_t n) {
    the_mem->memcpy_from_host(dest, src, n);
  }
};

// SMT: DO NOT need to pass in context
// load_elf_file is never invoked by a datapath that has a pointer
// to a pipelined-model. load_elf_file is always invoked by a datapath
// that has a pointer to an arch-model. no arch-model has more than
// one store buffer or memory space. all the references to "state"
// in this function are really references to the arch-model and its
// functions.
int
datapath::load_elf_file(const char* filename,
                        int argc,
                        char* argv[],
                        datapath_reg_init* dri)
{
  uint32_t start_pc;
  uint32_t start_freemem;
  uint32_t start_text_seg;
  uint32_t end_text_seg;
  elf_loader<memcpy_functor>::elf_return retval;
  memcpy_functor memcpy_func(state);
  elf_loader<memcpy_functor> the_loader;

  int errval = the_loader.read_elf_file(filename, &memcpy_func, &retval, false);
  if (errval != 0) return errval;

  start_pc = retval.start_pc;
  start_freemem = retval.start_freemem;
  start_text_seg = retval.start_segment;
  end_text_seg = retval.end_segment;

  // we put the arg vector at the start of the freemem
  // then pass arc and argv in regs 4 and 5
  // so that the _start procedure can find it.
  uint32_t string_size = 0;
  for (int i = 0; i < argc; i++) {
    string_size += (strlen(argv[i]) + 1); // don't forget terminating null!
  }

  uint32_t string_pos = start_freemem;
  uint32_t argv_pos = (string_pos + string_size + 15) & 0xfffffff0;

  // copy the string table
  for (int i = 0; i < argc; i++) {
    state->strcpy_from_host(string_pos, argv[i]);
    state->store_uint32(string_pos, argv_pos + (4 * i));
    string_pos += (strlen(argv[i]) + 1);
  }
  // and the final null:
  state->store_uint32(0, argv_pos + (4 * argc));

  // initialize pc, stackptr, and the argc, argv params to call the entry point:
  dri->start_text = start_text_seg;
  dri->end_text = end_text_seg;
  dri->pc = start_pc;
  dri->reg_29 = sign_extend64(0xfffffff0); // stack ptr
  dri->reg_4 = sign_extend64(argc);        // reg 4
  dri->reg_5 = sign_extend64(argv_pos);    // reg 5

#if 0
  for (uint32_t addr = start_text_seg;
       addr <= end_text_seg;
       addr += 4)
  {
    decoder::instr_h the_instr = decoder::run(state->load_uint32(addr));
    printf("%08x:\t", addr);
    pretty_print(the_instr);
    printf("\n");
  }
#endif

  return errval;
}
