#ifndef DEPENDENCY_WINDOW_H
#define DEPENDENCY_WINDOW_H

#include <inttypes.h>
#include <stdlib.h>
#include "circuit.h"
#include "decode.h"
#include "pool.h"

extern uint32_t debug_mask;
extern uint64_t cycle_count;
extern uint32_t superpipeline_factor;

enum dep_chain_e {
  DEP_CHAIN_S,
  DEP_CHAIN_T,
  DEP_CHAIN_NULL
};

class chain_object {
public:
  uint32_t reg_tag;
  dep_chain_e chain;
  uint32_t entry;

  chain_object(uint32_t x_reg_tag = 0, dep_chain_e x_c = DEP_CHAIN_NULL, uint32_t x_e = ~0) :
    reg_tag(x_reg_tag),
    chain(x_c),
    entry(x_e)
  { }

  bool operator==(const chain_object& rhs) const {
    return ((reg_tag == rhs.reg_tag) &&
            (chain == rhs.chain) &&
            (entry == rhs.entry));
  }

  void print() {
    printf("chain %u reg_tag %u entry %u\n", chain, reg_tag, entry);
  }

};

class instruction_object {
private:
  decoder::instr_h instr;
  chain_object s_chain;
  chain_object t_chain;
  bool s_ready, t_ready;  

public:
  instruction_object() :
    instr(decoder::noop),
    s_chain(0),
    t_chain(0),
    s_ready(true),
    t_ready(true)
    { }

  instruction_object(decoder::instr_h x_instr) :
    instr(x_instr),
    s_chain(x_instr->phys_s_reg),
    t_chain(x_instr->phys_t_reg),
    s_ready(true),
    t_ready(true)
    { }

  instruction_object(decoder::instr_h x_instr, bool x_s, bool x_t) :
    instr(x_instr),
    s_chain(x_instr->phys_s_reg),
    t_chain(x_instr->phys_t_reg),
    s_ready(x_s),
    t_ready(x_t)
    { }

  void print() {
    instr->print("instruction_object: ");
    printf("\n");
    s_chain.print();
    t_chain.print();
  }

  decoder::instr_h get_instr() { return instr; }
  const chain_object& get_s_chain() const { return s_chain; }
  const chain_object& get_t_chain() const { return t_chain; }

  void set_chain(dep_chain_e x_chain, const chain_object& x_next) {
    switch(x_chain) {
    case DEP_CHAIN_S:
      assert(s_chain.reg_tag == x_next.reg_tag);
      s_chain = x_next;
      s_ready = false;
      break;
    case DEP_CHAIN_T:
      assert(t_chain.reg_tag == x_next.reg_tag);
      t_chain = x_next;
      t_ready = false;
      break;
    default:
      assert(0 && "set_chain: setting invalid chain type");
      break;
    }
  }

  void set_chain_ready(dep_chain_e x_chain, bool x_val = true) {
    switch(x_chain) {
    case DEP_CHAIN_S:
      s_ready = x_val;
      break;
    case DEP_CHAIN_T:
      t_ready = x_val;
      break;
    default:
      assert(0 && "set_chain_ready: setting invalid chain type");
      break;
    }
  }

  bool get_chain_ready(dep_chain_e x_chain) const {
    switch(x_chain) {
    case DEP_CHAIN_S:
      return s_ready;
      break;
    case DEP_CHAIN_T:
      return t_ready;
      break;
    default:
      assert(0 && "set_chain_ready: setting invalid chain type");
      return false;
      break;
    }
    return false; //to keep gcc4.0 happy
  }

  bool registers_ready() const {
    return s_ready && t_ready;
  }

  uint32_t get_deps() const {
    //    printf("get deps %u %u = %u\n", !s_ready, !t_ready, !s_ready + !t_ready);
    return !s_ready + !t_ready;
  }

  const chain_object& get_chain(dep_chain_e x_chain) const {
    switch(x_chain) {
    case DEP_CHAIN_S:
      return s_chain;
      break;
    case DEP_CHAIN_T:
      return t_chain;
      break;
    default:
      assert(0 && "get_chain: getting invalid chain type");
      break;
    }
    assert(0); //you should never get here
    return t_chain; //to keep GCC 4.0 happy
  }

  ~instruction_object() {
    
  }
};

enum wakeup_result_e {
  WAKEUP_SUCCESS,
  WAKEUP_INSTR_NOT_READY,
  WAKEUP_EMPTY_CHAIN
};

class dependency_window_c : circuit {
  typedef map<uint32_t, chain_object> dep_chain_t;

private:
  pool_c<instruction_object> i_pool;
  dep_chain_t dep_chain_head;
  dep_chain_t dep_chain_tail;

  inline 
  void add_to_chain(const chain_object& x_tail) {
    if(x_tail.reg_tag) {
      // set the tail to being unready
      instruction_object& new_tail_obj = i_pool.get(x_tail.entry);
      new_tail_obj.set_chain_ready(x_tail.chain, false);

      if(debug_mask & 0x0008) {
        printf("IPOOL::add_to_chain>");
        new_tail_obj.print();
      }

      // add the new tail to the end of the dep chain
      if(dep_chain_tail.find(x_tail.reg_tag) != dep_chain_tail.end()) {
        const chain_object& dep_tail = dep_chain_tail[x_tail.reg_tag];
        instruction_object& tail_i_obj = i_pool.get(dep_tail.entry);

        tail_i_obj.set_chain(dep_tail.chain, x_tail);

        if(debug_mask & 0x0008) {
          printf("old tail:");
          tail_i_obj.print();
        }
      }
      else {
        assert(dep_chain_head.find(x_tail.reg_tag) == dep_chain_head.end());
        dep_chain_head[x_tail.reg_tag] = x_tail;
      }

      dep_chain_tail[x_tail.reg_tag] = x_tail;
    }
  }

public:
  dependency_window_c(uint32_t x_pool_size) :
    circuit(),
    i_pool(x_pool_size),
    dep_chain_head(),
    dep_chain_tail()
  { }

  void recalc() { }
  
  ~dependency_window_c() {

  }

  size_t num_free() {
    return i_pool.num_free();
  }

  size_t num_used() {
    return i_pool.num_used();
  }

  void print_info() {
    printf("dep_head_chain:\n");
    for(dep_chain_t::const_iterator itr = dep_chain_head.begin(); itr != dep_chain_head.end(); ++itr) {
      assert(itr->first == itr->second.reg_tag);
      printf("reg %u -> \n\tentry %u \n\t", itr->first, itr->second.entry);
      i_pool.get(itr->second.entry).get_instr()->print("instr:");
      printf("\n\tchain %u ready %u\n", itr->second.chain, i_pool.get(itr->second.entry).get_chain_ready(itr->second.chain));

    }
  }

//   typedef map<uint64_t, uint64_t> cancel_range_t;
//   bool inside_cancel_range(const uint64_t x_ts, const cancel_range_t& cancel_ranges) const {
//     // find the first range end point that is greater-than x_ts
//     cancel_range_t::const_iterator range_end = cancel_ranges.upper_bound(x_ts);
//     if(range_end != cancel_ranges.end()) {
//       if(debug_mask & 0x0008) printf("scheduler::inside_cancel_range %llu range: (%llu %llu)\n",
//                                      x_ts, range_end->second, range_end->first);
//       return (range_end->second <= x_ts);
//     }
//     else {
//       return false;
//     }
//   }

  vector<uint32_t> count_dependencies() {
    vector<uint32_t> ret_instr_dep_count(4,0);
    map<uint32_t, uint32_t> visited_instr_map;

    for(dep_chain_t::iterator itr = dep_chain_head.begin(); itr != dep_chain_head.end(); ++itr) {
      chain_object dep = itr->second;

      while(dep.chain != DEP_CHAIN_NULL) {
        instruction_object& i_obj = i_pool.get(dep.entry);

        if(visited_instr_map.find(dep.entry) == visited_instr_map.end()) {
          ++ret_instr_dep_count[i_obj.get_deps()];
          visited_instr_map[dep.entry] = true;
        }

        dep = i_obj.get_chain(dep.chain);
      }
    }
    return ret_instr_dep_count;
  }

  uint32_t get_chain_length(uint32_t x_reg) {
    uint32_t ret_val = 0;
    if(dep_chain_head.find(x_reg) != dep_chain_head.end()) {
      chain_object dep = dep_chain_head[x_reg];

      while(dep.chain != DEP_CHAIN_NULL) {
        instruction_object& i_obj = i_pool.get(dep.entry);
        ++ret_val;
        dep = i_obj.get_chain(dep.chain);
      }
    }
    return ret_val;
  }

  uint32_t num_sets() const {
    uint32_t head = dep_chain_head.size();
    assert(head == dep_chain_tail.size());
    return head;
  }

  void flush_canceled_instrs(bool (*cancel_function)(const uint32_t, const uint64_t), vector<pair<uint32_t, uint64_t> >& ret_reg_tags, vector<uint32_t>& ret_instr_dep_count) {
    // for counting dependencies
    map<uint32_t, uint32_t> remaining_instr_map;
    for(uint32_t i = 1; i < 4; ++i)
      ret_instr_dep_count[i] = 0;

    // walk the instruction list removing cancled instructions
    dep_chain_t::iterator itr;

    for(dep_chain_t::iterator next_itr = dep_chain_head.begin(); next_itr != dep_chain_head.end();) {
      // itr will be modified
      itr = next_itr++;
      chain_object dep = itr->second;
      chain_object prev_dep = dep;

      while(dep.chain != DEP_CHAIN_NULL) {
        instruction_object& i_obj = i_pool.get(dep.entry);
        if(debug_mask & 0x0008) {
          printf("cycle:%llu ", cycle_count);
          i_obj.get_instr()->print("flushtest: ");
          printf("\n");
        }

        // debug
        assert(!i_obj.get_chain_ready(dep.chain));
        assert(!i_obj.registers_ready());

        chain_object next_dep = i_obj.get_chain(dep.chain);
        if(cancel_function(i_obj.get_instr()->context, i_obj.get_instr()->instr_num)) {
          // remove the instruction from the current chain
          if(prev_dep == dep) {
            if(debug_mask & 0x0008) printf("--head of queue\n");
            // possible that this emptied this chain, check to be sure
            if(dep == dep_chain_tail[dep.reg_tag]) {
              if(debug_mask & 0x0008) printf("--dep chain should be empty\n");
              dep_chain_head.erase(dep.reg_tag);
              dep_chain_tail.erase(dep.reg_tag);
            }
            else {
              if(debug_mask & 0x0008) printf("--creating new head of chain\n");
              // new head of the dep chain (prev_dep and head are the same)
              prev_dep = i_obj.get_chain(dep.chain);
              dep_chain_head[dep.reg_tag] = prev_dep;
            }
          }
          else {
            if(debug_mask & 0x0008) printf("--instr not head of queue\n");
            // not removing instruction at the head
            instruction_object& prev_obj = i_pool.get(prev_dep.entry);
            prev_obj.set_chain(prev_dep.chain, i_obj.get_chain(dep.chain));

            // adjust tail ptr as needed
            if(dep == dep_chain_tail[dep.reg_tag]) {
              if(debug_mask & 0x0008) printf("--prev instr is new tail\n");
              dep_chain_tail[dep.reg_tag] = prev_dep;
            }
          }

          // set the chain as being ready
          i_obj.set_chain_ready(dep.chain);        

          // if the object is free from all chains, remove it
          if(i_obj.registers_ready()) {
            if(debug_mask & 0x0008) {
              printf("+++instr is removed, no deps\n");
            }
            decoder::instr_h instr = i_obj.get_instr();
            if(instr->phys_dest_reg != 0) {
              ret_reg_tags.push_back(make_pair(instr->phys_dest_reg,
                                              instr->instr_num));
            }

            i_pool.remove(dep.entry);
          }
        } // if(cancled)
        else {
          // update prev_dep to the most recent chain obj seen
          prev_dep = dep;
          // update the count of deps for instr that will remain in the window
          remaining_instr_map[dep.entry] = true;
        }          

        // next chain obj
        dep = next_dep;
      } // while(still deps in chain)
    } // for each dep chain

    for(map<uint32_t, uint32_t>::const_iterator itr = remaining_instr_map.begin();
        itr != remaining_instr_map.end(); ++itr) {
      instruction_object& i_obj = i_pool.get(itr->first);
      ++ret_instr_dep_count[i_obj.get_deps()];
    }

    // sanity check
    assert(num_used() == (ret_instr_dep_count[1] + 
                          ret_instr_dep_count[2] + 
                          ret_instr_dep_count[3]));
  }

  bool insert(decoder::instr_h x_instr, bool x_s_ready, bool x_t_ready) {
    if(debug_mask & 0x0008) {
      x_instr->print("IPOOL::insert> ");
      printf(" ready s:%u t:%u\n", x_s_ready, x_t_ready);
      printf("s_dep_len %u\n", get_chain_length(x_instr->phys_s_reg));
      printf("t_dep_len %u\n", get_chain_length(x_instr->phys_t_reg));
    }

    //    assert(x_t_ready);  // no more than two deps are allowed in this model

    bool s_t_unequal = (x_instr->phys_s_reg != x_instr->phys_t_reg);

    // before checking for conflicts and inserting instructions, pick which
    // list(s) to put the instruction on
    bool s_insert = !x_s_ready;
    bool t_insert = !x_t_ready && s_t_unequal;

    // insert instruction with no deps into the pool
    //   will fail if not enough room
    uint32_t entry = i_pool.insert(instruction_object(x_instr, x_s_ready, x_t_ready || !s_t_unequal)); 
    if(debug_mask & 0x0008) { printf("inserted into entry %u\n", entry); }
    // update the dep chains:
    if(s_insert) add_to_chain(chain_object(x_instr->phys_s_reg, DEP_CHAIN_S, entry));
    if(t_insert) add_to_chain(chain_object(x_instr->phys_t_reg, DEP_CHAIN_T, entry));

    return true;
  }

  // test for the possible success of removing the next instr on this set
  bool test_instr(uint32_t x_reg, decoder::instr_h& ret_instr) {
    if(debug_mask & 0x0008) printf("DEP_WIN::TEST_SUCCESS> %u ", x_reg);
    if(dep_chain_head.find(x_reg) != dep_chain_head.end()) {
      chain_object dep_head = dep_chain_head[x_reg];
      instruction_object& head_i_obj = i_pool.get(dep_head.entry);
      if(debug_mask & 0x0008) printf("deps %u\n", head_i_obj.get_deps());
      ret_instr = head_i_obj.get_instr();
      return true;
    }
    else {
      if(debug_mask & 0x0008) printf("--empty chain--\n");
      return false;
    }
  }

  void rotate(uint32_t x_reg) {
    if(dep_chain_head.find(x_reg) != dep_chain_head.end()) {
      chain_object dep_head = dep_chain_head[x_reg];
      instruction_object& head_i_obj = i_pool.get(dep_head.entry);

      // if the head is the tail, there is nothing to do,
      // otherwise move the head to the tail
      if(!(dep_head == dep_chain_tail[x_reg])) {
        // update the head of the chain
        dep_chain_head[x_reg] = head_i_obj.get_chain(dep_head.chain);
        // put the old head on the end of the chain
        head_i_obj.set_chain(dep_head.chain, chain_object(x_reg)/*null chain ptr*/);
        add_to_chain(dep_head);
      }
    }
  }

  // assume wakeup will call consume() on a particular
  // register tag until a chain_end is returned
  wakeup_result_e consume(uint32_t x_reg, uint64_t x_ts, decoder::instr_h& ret_instr, bool& ret_chain_end, uint32_t& ret_deps) {
    if(debug_mask & 0x0008) {
      print_info();
      printf("IPOOL::consume> reg %u\n", x_reg);
    }

    if(dep_chain_head.find(x_reg) != dep_chain_head.end()) {

      chain_object dep_head = dep_chain_head[x_reg];
      instruction_object& head_i_obj = i_pool.get(dep_head.entry);
     
      if(debug_mask & 0x0008) {
        head_i_obj.print();
      }

      // debug
      assert(!head_i_obj.get_chain_ready(dep_head.chain));
      assert(!head_i_obj.registers_ready());

      // get the instruction attempting awakening
      ret_instr = head_i_obj.get_instr();

      // reg could be from a later instruction (really, it's true, no lie)
      // ignore the wakeup if that is case, return noop so that
      // the register is thrown out, there's nothing to wakeup here
      assert(ret_instr->instr_num > x_ts);

      // mark the instruction chain reg as being being ready
      head_i_obj.set_chain_ready(dep_head.chain);

      // remove the head/tail entries if we have emptied the dep list,
      // otherwise set the next entry as the chain head
      if(dep_head == dep_chain_tail[x_reg]) {
        //debug
        assert(head_i_obj.get_chain(dep_head.chain).chain == DEP_CHAIN_NULL);
        assert(head_i_obj.get_chain(dep_head.chain).reg_tag == dep_head.reg_tag);
        assert(head_i_obj.get_chain(dep_head.chain).entry == (uint32_t)~0);

        dep_chain_head.erase(x_reg);
        dep_chain_tail.erase(x_reg);
        ret_chain_end = true;
      }
      else {
        // update the head of the chain
        dep_chain_head[x_reg] = head_i_obj.get_chain(dep_head.chain);
        ret_chain_end = false;
      }

      // return the number of remaining dependencies
      ret_deps = head_i_obj.get_deps();

      // return success
      bool ret_success = head_i_obj.registers_ready();

      // place the instruction back into the pool
      if(ret_success) i_pool.remove(dep_head.entry);

      if(ret_success) {
        if(debug_mask & 0x0008) printf("--wakeup success\n");
        return WAKEUP_SUCCESS;
      }
      else {
        if(debug_mask & 0x0008) printf("--wakeup instr not ready\n");
        return WAKEUP_INSTR_NOT_READY;
      }
    }
    else {
      if(debug_mask & 0x0008) printf("--wakeup empty chain\n");
      ret_instr = decoder::noop;
      ret_chain_end = true;
      ret_deps = 0;
      return WAKEUP_EMPTY_CHAIN;
    }
    return WAKEUP_INSTR_NOT_READY; //to make gcc4.0 happy
  }

  //  bool size();
  //  bool empty();
};

#endif
