#ifndef DECODE_H_GUARD
#define DECODE_H_GUARD

#include <vector>

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <bitset>
#include "gcp.h"
#include "bitops.h"
#include "long_vector.h"
#include "globals.h"

using namespace std;

namespace decoder
{

  enum opcodes {sll = 0, srl, sra, sllv, srlv, srav,
                dsll, dsrl, dsra, dsllv, dsrlv, dsrav,
                jr, jalr,
                mull, div, divu, mulh, mulhu,
                dmull, ddiv, ddivu, dmulh, dmulhu,
                addu, subu, uand, uor, uxor, nor,
                daddu, dsubu,
                slt, sltu,
                beq, bne,
                addiu, slti, sltiu, andi, ori, xori, aui,
                daddiu,
                lb, lh, lw, ld, lbu, lhu, lwu,
                sb, sh, sw, sd,
                bltz, blez, bgez, bgtz,
                j, jal,
                add_s, sub_s, mul_s, div_s, abs_s, neg_s,
                trunc_l_s, cvt_s_l, cvt_l_s, trunc_w_s, c_eq_s, c_lt_s, c_le_s,
                add_d, sub_d, mul_d, div_d, abs_d, neg_d,
                trunc_l_d, cvt_d_l, cvt_l_d, trunc_w_d, c_eq_d, c_lt_d, c_le_d,
                cvt_s_d, cvt_d_s,
                syscall_halt,
                syscall_unix,
                syscall_ftostr,
                syscall_strtof,
                signaling_noop};

  enum special_regs { REG_LINK = 31, REG_LO = 32, REG_HI = 33, REG_FD = 34 };


  namespace get_field
  {
    inline
    uint32_t
    opcode(uint32_t instr) { return (instr >> 26); }

    inline
    uint32_t
    op2(uint32_t instr) { return (instr & 0x1f); }



    inline
    uint32_t
    rs(uint32_t instr) { return ((instr >> 21) & 0x1f); }

    inline
    uint32_t
    rt(uint32_t instr) { return ((instr >> 16) & 0x1f); }

    inline
    uint32_t
    rd(uint32_t instr) { return ((instr >> 11) & 0x1f); }

    inline
    uint32_t
    shamt(uint32_t instr) { return ((instr >> 5) & 0x3f); }

    inline
    uint32_t
    unsigned_immediate(uint32_t instr) { return ((instr << 16) >> 16); }

    inline
    uint32_t
    jump_immediate(uint32_t instr) { return ((instr << 6) >> 6); }

    inline
    int32_t
    signed_immediate32(uint32_t instr) {
      // sign extend
      return (((int32_t)(instr << 16)) >> 16);
    }

    inline
    int64_t
    signed_immediate64(uint32_t instr) {
      // sign extend
      return (((int64_t)((int64_t)(instr) << 48)) >> 48);
    }
  };


  struct instr_rep {
  public:
  //default constructor
  instr_rep()       : 
    instr(0),
    opcode(sll),
    noop(true),
    arch_s_reg(0),
    arch_t_reg(0),
    arch_dest_reg(0),
    phys_s_reg(0),
    phys_t_reg(0),
    phys_dest_reg(0),
    bus_latency(1),
    is_load(false),
    is_store(false),
    is_syscall(false),
    is_branch(false),
    is_jump(false),
    is_link(false),
    is_indirect_jump(false),
    is_taken(false),
    next_pc(0),
    program_counter(0),
    predicted_pc(0),
    reorder_slot((uint32_t)(-1)),
    completed(false),
    decode_ts(0),
    dispatch_ts(0),
    issue_ts(0),
    completion_ts(0),
    canceled(false),
    ra_stack_ptr(0),
    ra_stack_head(0),
    wb_data(0ULL),
    valid(0), 
    vaddr(0),
    base_reg_val(0),
    instr_num(0),
    sb_slot(0),
    context(~0),
    rat_checkpt(0),
    stq_slot((size_t)(-1)),
    ldq_slot((size_t)(-1)),
    l1_hit(1),
    l2_hit(1),
    prev_pc(~0),
    mispredict(0) {}

    instr_rep(uint32_t i, opcodes set_opcode, bool is_noop = false)
      : instr(i),
        opcode(set_opcode),
        noop(is_noop),
        arch_s_reg(0),
        arch_t_reg(0),
        arch_dest_reg(0),
        phys_s_reg(0),
        phys_t_reg(0),
        phys_dest_reg(0),
        bus_latency(1),
        is_load(false),
        is_store(false),
        is_syscall(false),
        is_branch(false),
        is_jump(false),
        is_link(false),
        is_indirect_jump(false),
        is_taken(false),
        next_pc(0),
        program_counter(0),
        predicted_pc(0),
        reorder_slot((uint32_t)(-1)),
        completed(false),
        decode_ts(0),
        dispatch_ts(0),
        issue_ts(0),
        completion_ts(0),
        canceled(false),
        ra_stack_ptr(0),
        ra_stack_head(0),
        wb_data(0ULL),
        valid(0), 
        vaddr(0),
        base_reg_val(0),
        instr_num(0),
        sb_slot(0),
        context(~0),
        rat_checkpt(0),
	stq_slot((size_t)(-1)),
	ldq_slot((size_t)(-1)),
        l1_hit(1),
        l2_hit(1),
        prev_pc(~0),
        mispredict(0)
    { }

    uint32_t  instr;            // actual fetched bits
    opcodes   opcode;
    bool      noop;
    uint32_t  arch_s_reg;
    uint32_t  arch_t_reg;
    uint32_t  arch_dest_reg;
    uint32_t  phys_s_reg;
    uint32_t  phys_t_reg;
    uint32_t  phys_dest_reg;
    size_t    bus_latency; // expected writeback latency, this is just a prediction
    bool      is_load;
    bool      is_store;
    bool      is_syscall;
    bool      is_branch;
    bool      is_jump;
    bool      is_link;
    bool      is_indirect_jump;
    bool      is_taken;
    uint32_t  next_pc;          // next in trace
    uint32_t  program_counter;  // location fetched from
    uint32_t  predicted_pc;     // destination predicted by branch predictor
    uint32_t  reorder_slot;     // entry number in reo\rder buffer table
    bool      completed;
    uint32_t  decode_ts;
    uint64_t  dispatch_ts;
    uint64_t  issue_ts;
    uint64_t  completion_ts;
    bool      canceled;
    uint32_t  ra_stack_ptr;    // return address stack pointer at fetch time
    uint32_t  ra_stack_head;   // return address stack head at fetch time
    uint64_t  wb_data;
    bool      valid;            // data computed in exe is valid
                                // (all sources were available upon execute)
    uint32_t  vaddr;            // Memory address to store or load from.
    uint32_t  base_reg_val;
    uint64_t  instr_num;        // Unique instruction number
    size_t    sb_slot;          // Scoreboard Slot
    uint32_t  context;
   uint32_t  rat_checkpt;               //recovery checkpoint. for loads, its checkpoint before renaming. Otherwise, checkpoint after renaming
    size_t    stq_slot;
    size_t    ldq_slot;
    bool      l1_hit;          // mem_access info
    bool      l2_hit;
    uint32_t  prev_pc;         // tracks the pc that was fetched prior to this instruction
    bool      mispredict;      // whether or not a branch mispredicted, data is not filled in til
                               // the rob when the branch executes

    void print(const char* stagename) {
      printf("%s num %llu pc %08x a-s %u p-s %u a-t %u p-t %u a-d %u p-d %u comp %x canc %x ctext %d ratchk %u v %x lat %u nop %u",
             stagename,
             instr_num,
             program_counter,
             arch_s_reg,
             phys_s_reg,
             arch_t_reg,
             phys_t_reg,
             arch_dest_reg,
             phys_dest_reg,
             completed,
             canceled,
             context,
             rat_checkpt,
             valid,
             bus_latency,
             noop);
    }

    //is the instruction some kind of branch?
    bool is_ctrl_transfer_instr() {
      return (is_branch || is_jump || is_link || is_indirect_jump);
    }

    uint32_t unsigned_immediate() {
      return get_field::unsigned_immediate(instr);
    }
    int32_t signed_immediate32() {
      return get_field::signed_immediate32(instr);
    }
    int64_t signed_immediate64() {
      return get_field::signed_immediate64(instr);
    }
    uint32_t branch_immediate() {
      return signed_immediate32() * 4;
    }
    uint32_t jump_immediate() {
      return get_field::jump_immediate(instr) * 4;
    }
    uint32_t shamt_immediate() {
      return get_field::shamt(instr);
    }
    
  };

  typedef gcp<instr_rep> instr_h;

  namespace set_regs
  {
    inline
    void
    shift_imm(instr_h the_instr, uint32_t instr_bits)  {
      the_instr->arch_dest_reg = decoder::get_field::rd(instr_bits);
      the_instr->arch_t_reg = decoder::get_field::rt(instr_bits);
    }

    inline
    void
    rrr(instr_h the_instr, uint32_t instr_bits)  {
      the_instr->arch_dest_reg = decoder::get_field::rd(instr_bits);
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
      the_instr->arch_t_reg = decoder::get_field::rt(instr_bits);
    }

    inline
    void
    one_operand(instr_h the_instr, uint32_t instr_bits)    {
      the_instr->arch_dest_reg = decoder::get_field::rd(instr_bits);
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
    }

    inline
    void
    branch_store(instr_h the_instr, uint32_t instr_bits) {
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
      the_instr->arch_t_reg = decoder::get_field::rt(instr_bits);
    }

    inline
    void
    branch(instr_h the_instr, uint32_t instr_bits) {
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
    }

    inline
    void
    reg_imm(instr_h the_instr, uint32_t instr_bits) {
      the_instr->arch_dest_reg = decoder::get_field::rt(instr_bits);
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
    }

    inline

    void
    move_to_special(instr_h the_instr, uint32_t special, uint32_t instr_bits) {
      the_instr->arch_dest_reg = special;
      the_instr->arch_s_reg = decoder::get_field::rs(instr_bits);
    }

    inline
    void
    move_from_special(instr_h the_instr, uint32_t instr_bits, uint32_t special) {
      the_instr->arch_dest_reg = decoder::get_field::rd(instr_bits);
      the_instr->arch_s_reg = special;
    }
  };

  extern instr_h noop;

  instr_h
  decode_arith_map(uint32_t instr);


  instr_h
  decode_mul_map(uint32_t instr);

  instr_h
  decode_fpu_map(uint32_t instr);

  instr_h
  run(uint32_t instr);

};

#endif /* DECODE_H_GUARD */
