#ifndef ARCH_MODEL_H_GUARD
#define ARCH_MODEL_H_GUARD

#include <vector>
#include <deque>
#include "decode.h"
#include "execute.h"
#include "sparse_memory.h"
#include "pretty_print.h"
#include "globals.h"

class track_last_writer;

class arch_model : public state_semantics
{
public:
  // Listeners to model various performance or microarchitectural
  // state tracking info that only needs to be notified as each
  // individual instruction executes.  Use arch_trace's listeners if
  // you need to scan traces of instructions or manage queues of
  // instruction data.
  class listener {
  private:
    arch_model* the_arch_model;
  protected:
    const vector<uint64_t> model_reg_file() { return the_arch_model->reg_file; }
    const uint32_t model_pc() { return the_arch_model->the_pc; }
    sparse_memory* model_mem() { return the_arch_model->the_mem; }
  public:
    listener(arch_model* am) :
      the_arch_model(am) {
      am->register_listener(this);
    }
    virtual void notify(decoder::instr_h the_instr) = 0;
    virtual ~listener() { };
    friend class arch_model;
  };
  // listeners need permission to access private arch_model state:
  friend class listener;

private:
  vector< uint64_t > reg_file;
  uint32_t the_pc;
  sparse_memory* the_mem;
  datapath the_datapath;
  bool emulate_syscalls;

  std::vector<listener*> listeners;
  void notify_listeners(decoder::instr_h the_instr) {
    for (size_t i = 0; i < listeners.size(); i++) {
      listeners[i]->notify(the_instr);
    }
  }

  // FIXME: mif 2005-Nov-19: this is a total hack so we can expose the
  // last_writer_cpath_nums to the rat and rob in oracle mode
  track_last_writer* the_tlw_hack;

public:
  arch_model(sparse_memory* mem, bool em_syscalls) :
    state_semantics(),
    reg_file(32, 0),
    the_pc(99),
    the_mem(mem),
    the_datapath(this),
    emulate_syscalls(em_syscalls),
    listeners(),
    the_tlw_hack(0),
    halted(false)
    { 
    }

  void register_listener(listener* l) {
    listeners.push_back(l);
  }

  void register_last_writer_tracker(track_last_writer* tlw) {
    the_tlw_hack = tlw;
  }

  arch_model *make_copy(sparse_memory* mem, bool em_syscalls) {
    arch_model *temp = new arch_model(mem, em_syscalls);
//     sparse_memory * temp_mem;
//     if(!link_memories)
//       temp_mem = the_mem->make_copy();
//     else
//       temp_mem = new sparse_memory(the_mem);
//     arch_model *temp = new arch_model(temp_mem, em_syscalls);

    temp->reg_file = reg_file;
    temp->the_pc = the_pc;
    temp->halted = halted;
    return temp;
  }

  void print_sparse_memory_stats() {
    the_mem->print_stats();
  }

  void sync_mem_with_linked() {
    the_mem->sync_with_linked();
  }

  sparse_memory* get_sparse_mem_ptr() {
    return the_mem;
  }

  void replace_mem(const sparse_memory& m) {
    the_mem->replace_with(m);
  }

  uint32_t get_pc() {
    return the_pc;
  }

  //needed to repair state on retire_check_error
  vector<uint64_t> get_reg_file() {
    return reg_file;
  }

  // FIXME: mif 2005-Nov-19: hack needed for simpanics under oracle
  // polyflow renaming.  The code is actually in arch-listeners.cc
  // because arch-model.cc isn't actually the implementation file for
  // class arch_model!!!!!!!!!!!
  bool halted;

  uint64_t s_op64(decoder::instr_h instr) { return reg_file[instr->arch_s_reg]; }
  uint64_t t_op64(decoder::instr_h instr) { return reg_file[instr->arch_t_reg]; }
  uint64_t& dest(decoder::instr_h instr)  { return reg_file[instr->arch_dest_reg]; }
  uint64_t load_int8(uint32_t vaddr, uint32_t context=0) { return the_mem->load_int8(vaddr); }
  uint64_t load_uint8(uint32_t vaddr, uint32_t context=0) { return the_mem->load_uint8(vaddr); }
  uint64_t load_int16(uint32_t vaddr, uint32_t context=0) { return the_mem->load_int16(vaddr); }
  uint64_t load_uint16(uint32_t vaddr, uint32_t context=0) { return the_mem->load_uint16(vaddr); }
  uint64_t load_int32(uint32_t vaddr, uint32_t context=0) { return the_mem->load_int32(vaddr); }
  uint64_t load_uint32(uint32_t vaddr, uint32_t context=0) { return the_mem->load_uint32(vaddr); }
  uint64_t load_uint64(uint32_t vaddr, uint32_t context=0) { return the_mem->load_uint64(vaddr); }
  void store_uint8(uint8_t data, uint32_t vaddr, uint32_t context=0) { the_mem->store_uint8(data, vaddr); }
  void store_uint16(uint16_t data, uint32_t vaddr, uint32_t context=0) { the_mem->store_uint16(data, vaddr); }
  void store_uint32(uint32_t data, uint32_t vaddr, uint32_t context=0) { the_mem->store_uint32(data, vaddr); }
  void store_uint64(uint64_t data, uint32_t vaddr, uint32_t context=0) { the_mem->store_uint64(data, vaddr); }
  void memcpy_to_host(void* dest, const uint32_t src, size_t n, uint32_t context=0) { the_mem->memcpy_to_host(dest, src, n); }
  void memcpy_from_host(uint32_t dest, const void* src, size_t n, uint32_t context=0) { the_mem->memcpy_from_host(dest, src, n); }
  void strcpy_to_host(char* dest, const uint32_t src, uint32_t context=0) { the_mem->strcpy_to_host(dest, src); }
  void strcpy_from_host(uint32_t dest, const char* src, uint32_t context=0) { the_mem->strcpy_from_host(dest, src); }
  void emulate_syscall(uint32_t cmd_addr, uint32_t context=0) { 
    if(emulate_syscalls) {
      the_mem->emulate_syscall(cmd_addr); 
    }
  }
  uint32_t& program_counter() { return the_pc; }
  void zero_reg_0() { reg_file[0] = 0; }
  void print_err_msg(char *s) { fprintf(stderr, "%s", s); }

  decoder::instr_h do_cycle(FILE* tracefile = 0) {
    decoder::instr_h the_instr = decoder::run(the_mem->load_uint32(the_pc));

    the_instr->program_counter = the_pc;

    the_datapath.execute(the_instr);

    // post-exec knowledge
    if(the_pc != the_instr->program_counter+0x4)
      the_instr->is_taken = true;
    the_instr->next_pc = the_pc;

    notify_listeners(the_instr);

    return the_instr;
  }

  void halt() { halted = true; }

  int load_elf_file(const char* filename, int argc, char* argv[], datapath_reg_init* dri) {
    return the_datapath.load_elf_file(filename, argc, argv, dri);
  }

  void init_regs(datapath_reg_init* dri) {
    the_pc = dri->pc;
    reg_file[29] = dri->reg_29;
    reg_file[4] = dri->reg_4;
    reg_file[5] = dri->reg_5;
  }

  void set_emulate_syscalls(bool x_value) {
    emulate_syscalls = x_value;
  }

  ~arch_model() { 
    delete the_mem;
    for (size_t i = 0; i < listeners.size(); i++) {
      delete listeners[i];
    }
    listeners.clear();
  }
};

class call_packet {
public:
  uint32_t fn_addr;
  uint32_t call_addr;
  uint64_t instr_count;

  call_packet(uint32_t fa = 0, uint32_t ca = 0, uint64_t ic = 0) :
    fn_addr(fa),
    call_addr(ca),
    instr_count(ic) 
  { }
};

#endif /* ARCH_MODEL_H_GUARD */
