#include "arch-listeners.h"

track_last_writer::track_last_writer(arch_model* am) :
  arch_model::listener(am)
{
}

void
track_last_writer::notify(decoder::instr_h the_instr)
{

}


track_call_depth::track_call_depth(arch_model* am) :
  arch_model::listener(am),
  call_depth(0)
{
}

void
track_call_depth::notify(decoder::instr_h the_instr)
{
  //change call depth
  if( (the_instr->is_indirect_jump) && (the_instr->arch_s_reg==31) )
    call_depth --;            // return
  else if(the_instr->is_link)
    call_depth ++;
  assert(call_depth>=0);
}

spew_state_info::spew_state_info(arch_model* am,
                                 FILE* tf) :
  arch_model::listener(am),
  tracefile(tf)
{
}

void
spew_state_info::notify(decoder::instr_h the_instr)
{
  if (1)
  {
    uint32_t sp = model_reg_file()[29];
    uint32_t fp = model_reg_file()[30];

    fprintf(tracefile, "%08x:\t", model_pc()); // print the pc
    pretty_print(the_instr, tracefile); // pretty print the instruction
    fprintf(tracefile, "\n");

    // print the stack:
    for (uint32_t i = (uint32_t)-8; i > (sp - 32); i-=8) {
      if (i == sp) {
        fprintf(tracefile, ">");
      }
      else {
        fprintf(tracefile, " ");
      }
      if (i == fp) {
        fprintf(tracefile, "*");
      }
      else {
        fprintf(tracefile, " ");
      }
      fprintf(tracefile, "%08x: %08llx\n", i, model_mem()->load_uint64(i));
    }
  }

  // print the register file contents:
  for (int rreg = 0; rreg < 32; rreg+=4) {
    for (int reg = rreg; reg < (rreg + 4); reg++) {
      fprintf(tracefile, " %02d:%16llx", reg, model_reg_file()[reg]);
    }
    fprintf(tracefile, "\n");
  }
}

gen_cfg::gen_cfg(arch_model* am, uint32_t entry_pt) :
  arch_model::listener(am),
  the_count(),
  call_graph(),
  call_instrs(),
  entry_pt_pc(entry_pt)
{
}

void
gen_cfg::notify(decoder::instr_h the_instr)
{
  if (the_instr->is_link) {
    // the_instr is a call instruction:
    cfg_edge_t the_cfg_edge0(0, the_instr->next_pc); // begin_node -> procedure entry
    the_count[the_cfg_edge0] += 1;
    // calls always return to pc+4. however, pc+4 becomes a basic block leader now:
    cfg_edge_t the_cfg_edge1(the_instr->program_counter, the_instr->program_counter + 4);
    the_count[the_cfg_edge1] += 1;

    cfg_edge_t the_call_edge(the_instr->program_counter, the_instr->next_pc);
    call_graph[the_call_edge] += 1;
    call_instrs.insert(the_instr->program_counter); //the instruction at PC is a call
  }
  else {
    if ((the_instr->is_indirect_jump &&
         (the_instr->arch_s_reg == 31)) ||
        the_instr->opcode == decoder::syscall_halt) {
      // the_instr is a return point:
      cfg_edge_t the_cfg_edge(the_instr->program_counter, 0xffffffff);
      the_count[the_cfg_edge] += 1;
    }
    else {
      // "normal" instruction (including direct and indirect branches)
      cfg_edge_t the_edge(the_instr->program_counter, the_instr->next_pc);
      the_count[the_edge] += 1;
    }
  }
}

void
gen_cfg::print(FILE* cfg_file)
{
  typedef set<uint32_t> id_set_t;
  id_set_t targets;             // instr pcs we've seen already
  id_set_t joins;               // instr pcs we've seen more than once

  fprintf(cfg_file, "EntryPt %x\n", entry_pt_pc);

  // print all the control flow graph edges
  for (cfg_t::const_iterator i = the_count.begin();
       i != the_count.end();
       i++)
  {
    cfg_edge_t edge_id = i->first;
    uint32_t target = edge_id.second;
    fprintf(cfg_file, "cfg %x %x %llu\n",edge_id.first, edge_id.second,  i->second);
    if (targets.find(target) != targets.end()) {
      // if you're a target 'twice', you're a join
      joins.insert(target);
    }
    else {
      // make sure we record the target the first time we see it
      targets.insert(target);
    }
  }

  //print out the list of call instructions:
  for(node_set_t::const_iterator i = call_instrs.begin(); i!=call_instrs.end(); i++) {
    fprintf(cfg_file,"call %x\n", *i);
  }

  // print the call graph
  for (cfg_t::const_iterator i = call_graph.begin();
       i != call_graph.end();
       i++)
  {
    cfg_edge_t edge_id = i->first;
    uint32_t target = edge_id.second;
    fprintf(cfg_file, "callgraph %x %x %llu\n", edge_id.first, edge_id.second, i->second);
    if (targets.find(target) != targets.end()) {
      joins.insert(target);
    }
    else {
      targets.insert(target);
    }
  }
#if 0
  // print join points:
  for (cfg_t::const_iterator i = the_count.begin();
       i != the_count.end();
       i++)
  {
    cfg_edge_t edge_id = i->first;
    uint32_t target = edge_id.second;
    if (joins.find(target) != joins.end()) {
      fprintf(cfg_file, "cfg 0x%08x\t0x%08x\n", edge_id.first, edge_id.second);
    }
  }
  // print call joins (procs called from more than one place):
  for (cfg_t::const_iterator i = call_graph.begin();
       i != call_graph.end();
       i++)
  {
    cfg_edge_t edge_id = i->first;
    uint32_t target = edge_id.second;
    if (joins.find(target) != joins.end()) {
      fprintf(cfg_file, "call 0x%08x\t0x%08x\n", edge_id.first, edge_id.second);
    }
  }
#endif
}
