#ifndef STOREBUF_H_GUARD
#define STOREBUF_H_GUARD

#include <vector>
#include "packets.h"
#include "sparse_memory.h"

extern uint32_t debug_mask;

class store_buffer {
  vector<store_buffer_packet> the_buf;
  size_t head;
  size_t tail;
public:
  store_buffer(size_t size) :
    the_buf(size),
    head(0),
    tail(0)
    { }
  void clobber() {
    head = 0;
    tail = 0;
  }
  bool full() const {
    uint32_t num_avail = ((head + (the_buf.size() - (tail + 1))) % the_buf.size());
    return (num_avail <= 3);
  }
  bool empty() const {
    return (head == tail);
  }
  void insert(uint64_t instr_num, uint32_t address, uint64_t data, uint32_t size) {
    assert(!full());
    the_buf[tail].address = address;
    the_buf[tail].data = data;
    the_buf[tail].size = size;
    the_buf[tail].instr_num = instr_num;
    tail = (tail + 1) % the_buf.size();
  }
  void commit_head(sparse_memory *the_mem) {
    assert(!empty());
    the_buf[head].size = 0;
    head = (head + 1) % the_buf.size();
  }
  bool bypassed;
  uint64_t working_data;
  uint32_t working_vaddr;
  uint64_t working_instr_num;

  void* get_paddr(uint32_t vaddr) {
    uint32_t paddr = (uint32_t)(&working_data) + (vaddr & 0x7);
    return (void*)paddr;
  }
  void initiate_load(uint32_t vaddr, sparse_memory* the_mem, uint64_t instr_num) { 
    working_vaddr = vaddr & ~0x7;
    *(&working_data) = CVT_ENDIAN_DWORD(the_mem->load_uint64(working_vaddr));
    working_instr_num = instr_num;
  }
  void bypass_store(store_buffer_packet& the_store) {
    if (((the_store.address >> 3) == (working_vaddr >> 3)) && (the_store.instr_num < working_instr_num)) {
      void* my_paddr = get_paddr(the_store.address);
      switch (the_store.size) {
      case 1:
        *((uint8_t*)my_paddr) = (uint8_t)(the_store.data);
        break;
      case 2:
        *((uint16_t*)my_paddr) = CVT_ENDIAN_HWORD((uint16_t)(the_store.data));
        break;
      case 4:
        *((uint32_t*)my_paddr) = CVT_ENDIAN_WORD((uint32_t)(the_store.data));
        break;
      case 8:
        *((uint64_t*)my_paddr) = CVT_ENDIAN_DWORD(the_store.data);
        break;
      }
    }
  }
  void bypass_queue() {
    if(debug_mask & 0x800) {
      printf("bypassing load: (%llu)\n", working_instr_num);
      printf("bypass_queue> ");
    }
    // work through the entries in the queue, using each to modify the result
    for (size_t i = head; i != tail; i = (i + 1) % the_buf.size()) {
      if (debug_mask & 0x800)
        printf("(%x %llu) ", the_buf[i].address, the_buf[i].instr_num);
      bypass_store(the_buf[i]);
    }
  }

  uint64_t load_int8(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint8_t* my_paddr = (uint8_t*)get_paddr(vaddr);
    uint8_t data = *my_paddr;
    return (uint64_t)((int64_t)((int8_t)data));
  }
  uint64_t load_uint8(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint8_t* my_paddr = (uint8_t*)get_paddr(vaddr);
    uint8_t data = *my_paddr;
    return ((uint64_t)data);
  }
  uint64_t load_int16(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint16_t* my_paddr = (uint16_t*)get_paddr(vaddr);
    uint16_t memdata = *my_paddr;
    uint16_t data = CVT_ENDIAN_HWORD(memdata);
    return (uint64_t)((int64_t)((int16_t)data));
  }
  uint64_t load_uint16(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint16_t* my_paddr = (uint16_t*)get_paddr(vaddr);
    uint16_t memdata = *my_paddr;
    uint16_t data = CVT_ENDIAN_HWORD(memdata);
    return ((uint64_t)data);
  }
  uint64_t load_int32(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint32_t* my_paddr = (uint32_t*)get_paddr(vaddr);
    uint32_t memdata = *my_paddr;
    uint32_t data = CVT_ENDIAN_WORD(memdata);
    return (uint64_t)((int64_t)((int32_t)data));
  }
  uint64_t load_uint32(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint32_t* my_paddr = (uint32_t*)get_paddr(vaddr);
    uint32_t memdata = *my_paddr;
    uint32_t data = CVT_ENDIAN_WORD(memdata);
    return ((uint64_t)data);
  }
  uint64_t load_uint64(uint64_t instr_num, uint32_t vaddr, sparse_memory* the_mem) {
    initiate_load(vaddr, the_mem, instr_num);
    bypass_queue();
    uint64_t* my_paddr = (uint64_t*)get_paddr(vaddr);
    uint64_t memdata = *my_paddr;
    uint64_t data = CVT_ENDIAN_WORD(memdata);
    return data;
  }

};

#endif /* STOREBUF_H_GUARD */
