#ifndef SPARSE_MEMORY_H_GUARD
#define SPARSE_MEMORY_H_GUARD

#include <inttypes.h>
#include "sim_endian.h"
#include <stdio.h>
#include <vector>

using namespace std;

// command line variable
extern uint32_t cache_line_size;

#define CACHE_LINE_SIZE_BYTES 128 //in bytes
typedef vector<uint32_t> cacheline_t;
#define CACHE_LINE_INIT cacheline_t(CACHE_LINE_SIZE_BYTES/4,0)

class sparse_memory
{
private:
  // The memory will have a maximum size of 2^(PAGE_BITS + TABLE_BITS)
  // For reasonable performance you shouldn't use more than half that,
  // since the hashtable performance will degrade quickly.
  enum {
    PAGE_BITS = 15,
    PAGE_SIZE = (1 << PAGE_BITS),
    PAGE_MASK = (PAGE_SIZE - 1),
    TABLE_BITS = 15,             // Can NOT be larger than 16 for hashing to work
    TABLE_SIZE = (1 << TABLE_BITS),
    TABLE_MASK = (TABLE_SIZE - 1),
    MAGIC_NUMBER = 0x9e3779b9,
    EMPTY = (uint32_t)-1,
  };
  static uint32_t tag_of(uint32_t vaddr) {
    return (vaddr >> PAGE_BITS);
  }
  static uint32_t offset_of(uint32_t vaddr) {
    return (vaddr & ((1 << PAGE_BITS) - 1));
  }
  // we generate twice as many bits as we need:
  static uint32_t hash_of(uint32_t tag) {
    // This is from Knuth.  Basically we're multiplying by the golden
    // ratio, 0.61803398840129375. (Imagine that the radix point is
    // just to the left of the most significant bit.)
    return ((tag * MAGIC_NUMBER) >> (32 - (2 * TABLE_BITS)));
  }
  // we use the top half of the bits for the primary hash
  static uint32_t primary_hash_of(uint32_t hash0) {
    return (hash0 >> TABLE_BITS);
  }
  // and the bottome half we use for the secondary hash
  static uint32_t secondary_hash_of(uint32_t hash0) {
    // guarantee an odd number (so that the number is relatively prime
    // to TABLE_SIZE)
    return ((hash0 & TABLE_MASK) | 1);
  }

  struct entry_t {
    uint32_t tag;
    uint32_t paddr;
  };
  entry_t hashtab[TABLE_SIZE];
  size_t page_count;

  // Algorithm: use open addressing with double hashing (no chaining).
  // This is from Knuth (vol. 3, sec. 6.4, algorithm D).
  template <class Action>
  void* lookup(uint32_t vaddr) {
    uint32_t tag = tag_of(vaddr);
    int32_t hash0 = hash_of(tag);
    int32_t primary = primary_hash_of(hash0);
    int32_t i = primary;

    if (hashtab[i].tag == tag) return Action()(this, i, vaddr);

    int32_t secondary = secondary_hash_of(hash0);

  // secondary hash (after G. Knott)
    while (hashtab[i].tag != EMPTY) {
      i = ((i + secondary) & TABLE_MASK);
      if (hashtab[i].tag == tag) return Action()(this, i, vaddr);
    }

    // This page has never been accessed before.  So allocate a new page.

    if (page_count > (TABLE_SIZE / 2)) {
      if (page_count > ((TABLE_SIZE * 3) / 4)) {
        fprintf(stderr, "error: sparse_memory out of hash entries\n");
        abort();
      }
      else {
        fprintf(stderr, "warning: sparse_memory close to running out of hash entries, continuing anyway\n");
      }
    }

    hashtab[i].tag = tag;
    hashtab[i].paddr = (uint32_t) (new char[PAGE_SIZE]);

    page_count++;

    if (hashtab[i].paddr == 0) {
      fprintf(stderr, "OUT OF MEMORY IN imem::lookup()!!!!!!\n");
    }

    // initialize newly created pages to zero
    for(uint32_t p = 0; p < PAGE_SIZE; p++)
      *((char *)hashtab[i].paddr + p) = 0;

    return Action()(this, i, vaddr);
  }

  friend struct construct_address  {
    void* operator()(sparse_memory* mem,
                     uint32_t key,
                     uint32_t vaddr) {
      return (void*)(mem->hashtab[key].paddr + offset_of(vaddr));
    }
  };

  void* get_paddr(uint32_t vaddr) {
    return lookup<construct_address>(vaddr);
  }

public:
  sparse_memory() : page_count(0) {
    for (uint32_t i = 0; i < TABLE_SIZE; i++) {
      hashtab[i].tag = EMPTY;
    }
  }

  // for testing
  void print_table();

  uint64_t load_int8(uint32_t vaddr) {
    int8_t* paddr = (int8_t*)get_paddr(vaddr);
    uint8_t data = *paddr;
    return ((int64_t)((int8_t)data));
  }
  uint64_t load_uint8(uint32_t vaddr) {
    uint8_t* paddr = (uint8_t*)get_paddr(vaddr);
    uint8_t data = *paddr;
    return ((uint64_t)data);
  }
  uint64_t load_int16(uint32_t vaddr) {
    uint16_t* paddr = (uint16_t*)get_paddr(vaddr);
    uint16_t memdata = *paddr;
    uint16_t data = CVT_ENDIAN_HWORD(memdata);
    return ((int64_t)((int16_t)data));
  }
  uint64_t load_uint16(uint32_t vaddr) {
    uint16_t* paddr = (uint16_t*)get_paddr(vaddr);
    uint16_t memdata = *paddr;
    uint16_t data = CVT_ENDIAN_HWORD(memdata);
    return ((uint64_t)data);
  }
  uint64_t load_int32(uint32_t vaddr) {
    uint32_t* paddr = (uint32_t*)get_paddr(vaddr);
    uint32_t memdata = *paddr;
    uint32_t data = CVT_ENDIAN_WORD(memdata);
    return ((int64_t)((int32_t)data));
  }
  uint64_t load_uint32(uint32_t vaddr) {
    uint32_t* paddr = (uint32_t*)get_paddr(vaddr);
    uint32_t memdata = *paddr;
    uint32_t data = CVT_ENDIAN_WORD(memdata);
    return ((uint64_t)data);
  }
  uint64_t load_uint64(uint32_t vaddr) {
    uint64_t* paddr = (uint64_t*)get_paddr(vaddr);
    uint64_t memdata = *paddr;
    uint64_t data = CVT_ENDIAN_DWORD(memdata);
    return ((uint64_t)data);
  }
  void store_uint8(uint8_t data, uint32_t vaddr) {
    uint8_t* paddr = (uint8_t*)get_paddr(vaddr);
    *paddr = data;
  }
  void store_uint16(uint16_t data, uint32_t vaddr) {
    uint16_t* paddr = (uint16_t*)get_paddr(vaddr);
    uint16_t memdata = CVT_ENDIAN_HWORD(data);
    *paddr = memdata;
  }
  void store_uint32(uint32_t data, uint32_t vaddr) {
    uint32_t* paddr = (uint32_t*)get_paddr(vaddr);
    uint32_t memdata = CVT_ENDIAN_WORD(data);
    *paddr = memdata;
  }
  void store_uint64(uint64_t data, uint32_t vaddr) {
    uint64_t* paddr = (uint64_t*)get_paddr(vaddr);
    uint64_t memdata = CVT_ENDIAN_WORD(data);
    *paddr = memdata;
  }

  // routines for cache line loading and storing
  cacheline_t load_cacheline(uint32_t vaddr) {
    cacheline_t cline(cache_line_size/4, 0);
    for(uint32_t i=0; i < cache_line_size/4; i++) {
      cline[i] = load_uint32(vaddr + (4 * i));
    }
    return cline;
  }

  void store_cacheline(cacheline_t cline, uint32_t vaddr) {
    for(uint32_t i=0; i < cache_line_size/4; i++)
      {
	uint32_t* paddr = (uint32_t*)get_paddr(vaddr);
	*paddr = cline[i];
	vaddr = vaddr + 4;
      }
  }


  // routines to move data between host and simulator
  void memcpy_to_host(void* dest, const uint32_t src, size_t n);
  void memcpy_from_host(uint32_t dest, const void* src, size_t n);
  void strcpy_to_host(char* dest, const uint32_t src);
  void strcpy_from_host(uint32_t dest, const char* src);

  // syscall interface
  void emulate_syscall(uint32_t cmd_addr);
};

#endif /* SPARSE_MEMORY_H_GUARD */
