#include "sparse_memory.h"

using namespace std;

void
sparse_memory::print_table()
{
  for (unsigned i = 0; i < TABLE_SIZE; i++) {
    if (hashtab[i].tag != EMPTY) {
      printf("%x: %x -> %x\n", i, hashtab[i].tag * PAGE_SIZE, hashtab[i].paddr);
    }
  }
}

// These are all slow, but they should be correct, and they aren't on
// any critical paths.  Because we are copying between two different
// memories we don't need to worry about overlap.
void
sparse_memory::memcpy_to_host(void* dest, const uint32_t src, size_t n)
{
  char* output = (char*)dest;
  for (size_t i = 0; i < n; i++) {
    output[i] = load_uint8(src + i);
  }
}

void
sparse_memory::memcpy_from_host(uint32_t dest, const void* src, size_t n)
{
  char* input = (char*)src;
  for (size_t i = 0; i < n; i++) {
    store_uint8(input[i], dest + i);
  }
}

void
sparse_memory::strcpy_to_host(char* dest, const uint32_t src)
{
  uint8_t data;
  uint32_t s = src;
  do {
    data = load_uint8(s++);
    *dest = data;
    dest++;
  } while (data != 0);
  
}

void
sparse_memory::strcpy_from_host(uint32_t dest, const char* src)
{
  uint8_t data;
  do {
    data = *src;
    store_uint8(data, dest++);
    src++;
  } while (data != 0);
}

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

void
sparse_memory::emulate_syscall(uint32_t cmd_addr)
{
  uint32_t dispatch_code = load_uint32(cmd_addr);
  uint32_t arg1 = load_uint32(cmd_addr + 8);
  uint32_t arg2 = load_uint32(cmd_addr + 12);
  uint32_t arg3 = load_uint32(cmd_addr + 16);
  uint32_t result = (uint32_t)-1;

  fflush(stdout);
  fflush(stderr);

  switch (dispatch_code) {
  case 0x80: {
    char str[4096];
    int flags = 0;
    strcpy_to_host(str, arg1);
    if (arg2 & 0x1) {
      flags |= O_WRONLY;
    }
    if (arg2 & 0x0200) {
      flags |= O_CREAT;
    }
    if (arg2 & 0x0400) {
      flags |= O_TRUNC;
    }
    result = open(str, flags, S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP);
    break;
  }
  case 0x81: {
    result = close(arg1);
    break;
  }
  case 0x82: {
    char buf[arg3];
    result = read(arg1, buf, arg3);
    memcpy_from_host(arg2, buf, arg3);
    break;
  }
  case 0x83: {
    char buf[arg3];
    memcpy_to_host(buf, arg2, arg3);
    result = write(arg1, buf, arg3);
    break;
  }
  case 0x84: {
    result = lseek(arg1, arg2, arg3);
    break;
  }
  case 0x85: {
    char str[4096];
    strcpy_to_host(str, arg1);
    result = unlink(str);
    break;
  }
  case 0x86: {
    char str1[4096];
    char str2[4096];
    strcpy_to_host(str1, arg1);
    strcpy_to_host(str2, arg2);
    result = link(str1, str2);
    break;
  }
  case 0x87: {
    char str[4096];
    struct stat stat_obj;
    strcpy_to_host(str, arg1);
    result = stat(str, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x88: {
    struct stat stat_obj;
    result = fstat(arg1, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x89: {
    char str[4096];
    struct stat stat_obj;
    strcpy_to_host(str, arg1);
    result = lstat(str, &stat_obj);
    memcpy_from_host(arg2, &stat_obj, sizeof(struct stat));
    break;
  }
  case 0x90: {
    char str[4096];
    strcpy_to_host(str, arg1);
    result = chmod(str, arg2);
    break;
  }
  case 0x91: {
    char str1[4096];
    char str2[4096];
    strcpy_to_host(str1, arg1);
    strcpy_to_host(str2, arg2);
    result = rename(str1, str2);
  }
  default: 	{
    fprintf(stderr, "bogus unix syscall %x\n", dispatch_code);
  }
  }
  store_uint32(result, cmd_addr + 4);
}
