#ifndef SCOREBOARD_H
#define SCOREBOARD_H

#include <vector>

#include "circuit.h"
#include "regfile_circuit.h"
#include "decode.h"
#include "pretty_print.h"

using namespace std;

// Global variable potentially defined on command line
extern uint32_t num_phys_regs;
extern uint32_t scoreboard_fifo_size;
extern bool branch_speculate;      // defaults to false, command line might set true

// Global debug mask
extern uint32_t debug_mask;

class scoreboard : circuit
{
  // fifo of already fetched/decoded/renamed instructions:
  vector< decoder::instr_h > instr_buf;
  vector< bool > issued;
  vector< bool > completed;
  vector< bool > mispredicted;
  size_t head;
  size_t tail;

  bool fifo_almost_full() { 
    // Check to see how many fifo entries are available, a stall will occur
    // if there are 2 or less entries available
    uint32_t num_avail_fifo = 
      (head + (scoreboard_fifo_size - (tail + 1))) % scoreboard_fifo_size;

    // Fifo is almost full OR wait for retirement of instruction before branch
    // recovery
    return ((num_avail_fifo <= 2) || rob_stall());
  }

  bool fifo_empty() {
    return (head == tail);
  }

  bool safe_instr(decoder::instr_h instr) {
    if (instr->is_syscall)
    {
      return (rob_head() == instr->reorder_slot);
    }
    else {
      return true;              // everything else is "safe"
    }
  }

  void issue_instr(size_t slot, decoder::instr_h the_instr) {
    instr_out = the_instr;
    issued[slot] = true;
  }

  void recalc() {

    if (debug_mask & 0x1) {
      printf("head %d, tail %d (%d)\n", head, tail, fifo_almost_full());
      printf("%d\n", scoreboard_fifo_size);
      instr_buf[head]->print("SCRBRD>");
      printf("\n");
      pretty_print(instr_buf[head]);
      printf("\n");
    }

    if (branch_mispredict().mispredicted_branch()) {
      // reinitialize scoreboard as empty:
      head = 0;
      tail = 0;
      for (size_t i = 0; i < instr_buf.size(); i++) {
        instr_buf[i] = decoder::noop;
      }
      for (size_t i = 0; i < issued.size(); i++) {
        issued[i] = false;
	completed[i] = false;
        mispredicted[i] = false;
      }
      instr_out = decoder::noop;
      scoreboard_stall = false;
    }
    else {

      // mark completed instruction
      if (writeback_bus().reorder_slot != (size_t)-1) {
	size_t slot = writeback_bus().sb_slot;
        completed[slot] = true;
      }
      
      if (branch_bus().is_branch) {
	size_t slot = branch_bus().sb_slot;
        // jals and jalrs get completed by writeback_bus, not branch_bus
        if (instr_buf[slot]->phys_dest_reg == 0) {
          completed[slot] = true;
        }
        if (branch_bus().mispredict) {
          mispredicted[slot] = true;
        }
      }
      
      if (store_bus().is_store) {
	size_t slot = store_bus().sb_slot;
        completed[slot] = true;  
      }

      instr_out = decoder::noop;

      // find an instruction to issue, if possible
      bool unissued_store = false;
      bool unissued_load = false;
      vector<bool> reserved(num_phys_regs, false); // regs that haven't been read yet
      vector<bool> reg_avail(num_phys_regs, true); // regs that are complete
      for (size_t i = head; i != tail; i = (i + 1) % scoreboard_fifo_size) {
        decoder::instr_h the_instr = instr_buf[i];

        if (!(issued[i])) {

          if (reg_avail[the_instr->phys_dest_reg] && // no output dep
              !reserved[the_instr->phys_dest_reg] && // no anti dep
              reg_avail[the_instr->phys_s_reg] && // no true dep on s reg
              reg_avail[the_instr->phys_t_reg] && // no true dep on t reg
              !(the_instr->is_store && (unissued_store || unissued_load)) && // no stores out of order
              !(the_instr->is_load && unissued_store) && // no loads out of order
              safe_instr(the_instr)) // no speculative syscalls
          {
            issue_instr(i, the_instr); // found one!
            break;
          }

          if (debug_mask & 0x1) {
            the_instr->print("NOISSUE>");
            printf("\n\t\t");
            pretty_print(the_instr);
            printf("\n\t\t");
            for (size_t jj = 0; jj != reserved.size(); jj++) {
              bool r = reserved[jj];
              if ((jj & 0x7) == 0) printf(" ");
              printf("%d", r);
            }
            printf("\n\t\t");
            for (size_t jj = 0; jj != reg_avail.size(); jj++) {
              bool r = reg_avail[jj];
              if ((jj & 0x7) == 0) printf(" ");
              printf("%d", r);
            }
            printf("\n");
            printf("\t\tis-%d il-%d us-%d ul-%d safe-%d\n",
                   the_instr->is_store,
                   the_instr->is_load,
                   unissued_store,
                   unissued_load,
                   safe_instr(the_instr));
          }

          if (the_instr->is_store) {
            unissued_store = true;
          }
          if (the_instr->is_load) {
            unissued_load = true;
          }
          reserved[the_instr->phys_s_reg] = true;
          reserved[the_instr->phys_t_reg] = true;
        }

        if (!(completed[i])) {
          reg_avail[the_instr->phys_dest_reg] = false;
        }

        if (the_instr->is_syscall) {
          break;                // can't execute past a syscall
        }

        if (mispredicted[i]) {
          break;                // don't execute past a mispredicted branch
        }

        if ((!branch_speculate) && (the_instr->is_branch || the_instr->is_jump) && !completed[i]) {
          break;                // don't speculatively execute past a branch if inorder
        }

        reserved[0] = false;
        reg_avail[0] = true;
      }

      // if rename gave us an instruction we better remember it
      if (!instr_in()->noop) {
        decoder::instr_h mod_instr = instr_in();
        mod_instr->reorder_slot = rob_tail();
	mod_instr->sb_slot = tail;
        instr_buf[tail] = mod_instr;
        issued[tail] = false;
	completed[tail] = false;
        mispredicted[tail] = false;
        tail = (tail + 1) % scoreboard_fifo_size;
      }

      // release completed instrs from head of queue
      if(issued[head] && completed[head] && !mispredicted[head]) {
	issued[head] = false;
	completed[head] = false;
        mispredicted[head] = false;
	head = (head + 1) % scoreboard_fifo_size;
      }
	
      scoreboard_stall = fifo_almost_full();
    }

  }

public:
  scoreboard() :
    circuit(),

    instr_buf(scoreboard_fifo_size, decoder::noop),
    issued(scoreboard_fifo_size, false),
    completed(scoreboard_fifo_size, false),
    mispredicted(scoreboard_fifo_size, false),
    head(0),
    tail(0),
    
    instr_in(),
    rob_stall(),
    rob_tail(),

    writeback_bus(),
    store_bus(),
    branch_bus(),

    instr_out(decoder::noop),
    scoreboard_stall(false)
    {
    }

  // input from rename:
  inport<decoder::instr_h> instr_in;
  // inputs from rob:
  inport<bool> rob_stall;
  inport<size_t> rob_tail;
  inport<size_t> rob_head;      // ptr to instruction ready to commit
  inport<branch_packet> branch_mispredict; // branch mispredict throws out all instructions in scoreboard
  // inputs from exec_unit (detect instruction completion
  inport<bus_packet> writeback_bus;
  inport<store_packet> store_bus;
  inport<branch_packet> branch_bus;

  // outputs:
  statereg<decoder::instr_h> instr_out;
  statereg<bool> scoreboard_stall;
};

#endif /* SCOREBOARD_H */
