#ifndef ROB_H
#define ROB_H

#include "circuit.h"
#include "regfile_circuit.h"
#include "decode.h"
#include "packets.h"
#include "arch-model.h"

#include "pretty_print.h"

using namespace std;

// Global variable potentially defined on command line
extern uint32_t num_phys_regs;
extern uint32_t rob_size;

// Global debug variables
extern uint32_t debug_mask;
extern bool trace_on;
extern bool arch_compare_on;
extern uint32_t tornado_warning;
extern uint64_t cycle_count;

// the rob counts retired instructions
uint64_t the_instr_count = 0;
uint64_t the_retired_branch_count = 0;
uint64_t the_mispredicted_branch_count = 0;

class rob : circuit
{
  // fifo of already fetched/decoded/renamed instructions:
  register_file< decoder::instr_h> reorder_buf;
  register_file< branch_packet > branch_buf;
  statereg<size_t> recovery_tail;	// For debuging visualization only
  statereg<bool> branch_recovery;
  statereg<bool> cancelled_instr_rec;
  statereg<bool> instruction_retiring;
  branch_packet no_branch;

  //debug info
  uint32_t idle_cycles;
  // debug arch-model
  arch_model* the_arch_model;

  bool stall_prev_stages() {
    // Check to see how many fifo entries are available, a stall will occur
    // if there are 3 or less entries available
    uint32_t num_avail = 
      (rob_head() + (rob_size - (rob_tail() + 1))) % rob_size;
    // Fifo is almost full?
    return (num_avail <= 3);
  }

  bool fifo_empty() {
    return (rob_head() == rob_tail());
  }

  void recalc() {

    if(debug_mask & 0x0010) {
      int tail_ptr = rob_tail() - 1;
      if(tail_ptr < 0)
        tail_ptr = rob_size - 1;
      reorder_buf[tail_ptr]()->print("ROB IN>");
      printf(" hd %u tl %u rect %u stall %x\n", 
             rob_head(), rob_tail(), recovery_tail(), rob_stall());
      reorder_buf[rob_head()]()->print("ROBRET>");
      printf(" is cancelled? %d", reorder_buf[rob_head()]()->canceled);;
      printf(" brec %x rreg %u ret %x freelst %x\n", branch_recovery(), 
             retiring_instr()->arch_dest_reg, instruction_retiring(),
             cancelled_instr_rec());
      branch_buf[rob_head()]().print();
    }

    if(debug_mask & 0x0400) {
      printf("ROBTBL>");
      for(uint32_t i = 0; i < rob_size; i++) {
        if(i > 0 && (i % 32 == 0))
          printf("\n       ");
	
        if(i == rob_head())
          printf(" < %u", reorder_buf[i]()->phys_dest_reg);
        else if(i == rob_tail())
          printf(" > %u", reorder_buf[i]()->phys_dest_reg);
        else if((i == recovery_tail()) && (recovery_tail() != rob_tail()))
          printf(" >> %u", reorder_buf[i]()->phys_dest_reg);
        else
          printf(" %u", reorder_buf[i]()->phys_dest_reg);
      }
      printf("\n");
    }

    branch_out = no_branch; 
    retiring_instr = decoder::noop;
    branch_recovery = false;
    cancelled_instr_rec = false;
    instruction_retiring = false;
    commit_head_store = false;

    // insert instructions from previous stage into tail of queue:
    if (!instr_in()->noop) {
      decoder::instr_h the_instr = instr_in();
      if (branch_out().mispredicted_branch())
      {
        the_instr->completed = true;
        the_instr->canceled = true;
      }
      reorder_buf[rob_tail()] = the_instr;
      branch_buf[rob_tail()] = no_branch;
      rob_tail = (rob_tail() + 1) % rob_size;
    }

    if (!branch_out().mispredicted_branch()) {
      // retire completed instructions from head of queue:
      if (!fifo_empty()) {
        if (reorder_buf[rob_head()]()->completed) {

          if (!reorder_buf[rob_head()]()->canceled) {
	    instruction_retiring = true;
	    branch_out = branch_buf[rob_head()]();
	    if (branch_buf[rob_head()]().is_branch) {
	      the_retired_branch_count++;
	      if (branch_buf[rob_head()]().mispredict) {
		the_mispredicted_branch_count++;
	      }
	    }

	    if (reorder_buf[rob_head()]()->is_store) {
	      commit_head_store = true;
	    }	

	    the_instr_count++;

	    if(trace_on) {
	      printf("%x\n", reorder_buf[rob_head()]()->program_counter);
	      fflush(stdout);
	    }

	    // compare results with the arch_model
	    if(arch_compare_on) {

	      bool arch_problem = false;

	      decoder::instr_h arch_instr;
	      do {
		arch_instr = the_arch_model->do_cycle();
	      } while(!arch_instr->instr);

              if (debug_mask & 0x1000) {
                printf("retiring %08x\t%08x\t%x\n",
                       arch_instr->program_counter,
                       arch_instr->instr,
                       arch_instr->opcode);
                arch_instr->print("\nARCH_RETIRE>");
                reorder_buf[rob_head()]()->print("\nPIPE_RETIRE>");
                printf("\n");
                branch_buf[rob_head()]().print();
                pretty_print(arch_instr);
                printf("\n");
                fflush(stdout);
              }


	      if(arch_instr->program_counter != reorder_buf[rob_head()]()->program_counter) {
		printf("ARCH_COMPARE> MISMATCHED INSTRUCTION PC\n");
		arch_problem = true;
	      }
	      else {
		if((arch_instr->arch_dest_reg != 0) || reorder_buf[rob_head()]()->is_store) {
		  if(arch_instr->wb_data != reorder_buf[rob_head()]()->wb_data) {
		    printf("ARCH_COMPARE> INCORRECT WB_DATA: ");
		    printf("$%d <- %x", arch_instr->arch_dest_reg
			   , arch_instr->wb_data);
		    printf(" VS. ROB: $%d <- %x\n",
		       reorder_buf[rob_head()]()->arch_dest_reg,
		       reorder_buf[rob_head()]()->wb_data);
		    arch_problem = true;
		  }
		}
	      }

	      if(arch_problem) {
		arch_instr->print("ARCH_INSTR>");
		printf("\n");
		reorder_buf[rob_head()]()->print("ROBRET_INSTR>");
		printf("\n");
		printf("Cycle: %llu\n", cycle_count);
		exit(1);
	      }
	    }

	    if (debug_mask & 0x2000) {
	      if (reorder_buf[rob_head()]()->arch_dest_reg != 0) {
		printf("$%d <- %x\n",
		       reorder_buf[rob_head()]()->arch_dest_reg,
		       reorder_buf[rob_head()]()->wb_data);
	      }
	    }
	  }

	  else {
	    cancelled_instr_rec = true;
	  }
					
          // Recover regs to freelist from retired and canceled instructions
          if(reorder_buf[rob_head()]()->arch_dest_reg != 0) {
            retiring_instr = reorder_buf[rob_head()]();
          }

          if(rob_head() == recovery_tail())
            recovery_tail = (recovery_tail() + 1) % rob_size;

          rob_head = (rob_head() + 1) % rob_size;

	  if(tornado_warning != 0)
	    idle_cycles = 0;
        }
	else { // detect idle cycles
	  if(tornado_warning != 0) {
	    idle_cycles++;

	    if(idle_cycles > tornado_warning) {
	      printf("ROB LOCKUP DETECTED AT CYCLE: %llu\n", cycle_count);
	      reorder_buf[rob_head()]()->print("ROBHEAD>");
	      printf("\n");
	      tornado_warning = 0;
	      exit(1);
	    }
	  }
	}
      }
    }
    else {
      // last instruction was a branch mispredict: 
      rob_stall = false;
      branch_recovery = true; // For debugging visualization only
      idle_cycles = 0;

      // cancel all the instructions between head and tail:
      size_t position = rob_head();
      while (position != rob_tail()) {
        decoder::instr_h the_instr = reorder_buf[position]();
        the_instr->completed = true;
        the_instr->canceled = true;
        reorder_buf[position] = the_instr;
        // canceled branches should act like noops:
        branch_packet the_br_pack = branch_buf[position]();
        the_br_pack.is_branch = false;
        the_br_pack.taken = false;
        the_br_pack.mispredict = false;
        branch_buf[position] = the_br_pack;
        position = (position + 1) % rob_size;
      }

      // For debugging visualization - not functionally used
      if(rob_tail() > 0)
        recovery_tail = rob_tail() - 1;
      else
        recovery_tail = rob_size - 1;
    }

    // mark any instructions that were completed by execution unit:
    if (writeback_bus().reorder_slot != (size_t)-1) {
      decoder::instr_h the_instr = reorder_buf[writeback_bus().reorder_slot]();
      the_instr->completed = true;
      uint32_t wb_phys_reg = writeback_bus().reg_tag;
      assert(wb_phys_reg == the_instr->phys_dest_reg);
      the_instr->wb_data = writeback_bus().data;
      reorder_buf[writeback_bus().reorder_slot] = the_instr;
    }

    // mark any branches that were completed by execution unit:
    if (branch_bus().is_branch) {
      assert(branch_bus().reorder_slot != (size_t)-1);
      decoder::instr_h the_instr = reorder_buf[branch_bus().reorder_slot]();
      // complete the instruction only if it is *not* a jal or jalr:
      // jals and jalrs complete when they get a signal from the
      // writeback bus.
      if (the_instr->arch_dest_reg == 0) {
        the_instr->completed = true;
        reorder_buf[branch_bus().reorder_slot] = the_instr;
      }
      if (! the_instr->canceled) {
        branch_buf[branch_bus().reorder_slot] = branch_bus();
      }
    }

    // mark any store that has been buffered by the execution unit:
    if (store_bus().is_store) {
      assert(store_bus().reorder_slot != (size_t)-1);
      decoder::instr_h the_instr = reorder_buf[store_bus().reorder_slot]();
      the_instr->completed = true;
      // debugging purposes only:
      the_instr->wb_data = store_bus().data;
      reorder_buf[store_bus().reorder_slot] = the_instr;
    }

    rob_stall = stall_prev_stages();

  }


public:
  rob(arch_model* am) :
    circuit(),

    reorder_buf(rob_size, decoder::noop),
    branch_buf(rob_size, branch_packet()),
    branch_recovery(false),
    cancelled_instr_rec(false),
    instruction_retiring(false),
    no_branch(branch_packet()),
    idle_cycles(0),
    the_arch_model(am),
    
    instr_in(),
    branch_bus(),
    store_bus(),
    writeback_bus(),

    rob_head(0),
    retiring_instr(decoder::noop),
    branch_out(),
    rob_stall(false),
    rob_tail(0)
    { }

  // input from rename:
  inport<decoder::instr_h> instr_in;

  inport<branch_packet> branch_bus; // execution unit results for branches
  inport<store_packet> store_bus;
  inport<bus_packet> writeback_bus; // execution unit results for everything else

  // outputs:
  statereg<size_t> rob_head;
  statereg<decoder::instr_h> retiring_instr;
  statereg<branch_packet> branch_out;
  statereg<bool> rob_stall;
  statereg<size_t> rob_tail;
  statereg<bool> commit_head_store;
};

#endif /* ROB_H */
