#ifndef RENAME_STAGE_H
#define RENAME_STAGE_H

#include "circuit.h"
#include "regfile_circuit.h"
#include "packets.h"

#define NUM_ARCH_REGS 32

using namespace std;

//Global variable potentially defined on commandline, default is 64
extern uint32_t num_phys_regs;

class rename_stage : circuit
{
#if !defined(NORENAME)
  vector< uint32_t > current_RAT;
  vector< uint32_t > retirement_RAT;
  size_t sizeof_free_list;
  vector< uint32_t > free_list;
  size_t free_list_ptr;
  // free_list_ptr points just *above* the next free reg =>
  // pop:  x = free_list[--ptr]
  // push: free_list[ptr++] = x
#endif

  void recalc() {
    if (debug_mask & 0x0004) {
      instr_out()->print("RENAME>");
      printf(" rstll %x \n", rename_stall());
    }

    // THIS PART OF THE CODE HANDLES THE INSTRUCTION ARRIVING FROM THE DECODER:
    if (branch_mispredict().mispredicted_branch() ||
        scoreboard_stall() ||
        rename_stall() ||
        instr_in()->noop ||
        rob_stall()) {
      // flush the arriving instruction
      instr_out = decoder::noop;
    }
    else {
      // rename the arriving instruction
      decoder::instr_h the_instr = instr_in();

#if defined(NORENAME)
      // for an inorder machine, don't need true renaming:
      the_instr->phys_t_reg = the_instr->arch_t_reg;
      the_instr->phys_s_reg = the_instr->arch_s_reg;
      the_instr->phys_dest_reg = the_instr->arch_dest_reg;
#else
      the_instr->phys_t_reg = current_RAT[the_instr->arch_t_reg];
      the_instr->phys_s_reg = current_RAT[the_instr->arch_s_reg];
      if (the_instr->arch_dest_reg == 0) {
        the_instr->phys_dest_reg = 0;
      }
      else {
        the_instr->phys_dest_reg = free_list[--free_list_ptr];
        current_RAT[the_instr->arch_dest_reg] = the_instr->phys_dest_reg;
      }
#endif /* NORENAME */

      // write the renamed instruction to the output (goes to scoreboard and rob)
      instr_out = the_instr;
    }

    // THIS PART OF THE CODE HANDLES THE INSTRUCTION THAT IS RETIRING FROM THE ROB
    if (retiring_instr()->completed) {
      if (retiring_instr()->canceled) {
        // rob is telling us about a cancelled instruction (one that is being flushed because of a mispredict)

#if !defined(NORENAME)
        // instruction was cancelled: just free its physical destination regs
        if (retiring_instr()->phys_dest_reg != 0) {
          free_list[free_list_ptr++] = retiring_instr()->phys_dest_reg;
        }
#endif /* NORENAME */        
      }
      else {
        // instruction completed and is not canceled: normal retirement

#if !defined(NORENAME)
        // instruction completed and is not canceled: normal retirement
        if (retiring_instr()->arch_dest_reg != 0) {
          free_list[free_list_ptr++] = retirement_RAT[retiring_instr()->arch_dest_reg];
          retirement_RAT[retiring_instr()->arch_dest_reg] = retiring_instr()->phys_dest_reg;
        }
#endif /* NORENAME */        
      }
    }

    // THIS PART OF THE CODE HANDLES RENAMER STATE CHANGES THAT OCCUR
    // BECAUSE OF A MISPREDICT.  NOTE: Jump-and-link instructions (jal
    // and jalr) can both mispredict and write back to a register
    // (always architectural register 31).  Thus, during a cycle that
    // a mispredicted jump-and-link retires both
    // branch_mispredict.mispredicted_branch and
    // retiring_instr->completed will be true.
    if (branch_mispredict().mispredicted_branch()) {
#if !defined(NORENAME)
      // copy the retirement rat over the current rat
      for(uint32_t i = 0; i < NUM_ARCH_REGS; i++) {
        current_RAT[i] = retirement_RAT[i];
      }
#endif
    }

    // THE RENAMER STALLS WHEN IT GETS CLOSE TO RUNNING OUT OF
    // REGISTERS IN ITS FREELIST.  THIS SIGNAL TELLS THE FETCH AND
    // DECODE UNITS TO STALL.  NOTE THAT THIS SIGNAL DOESN'T GET TO
    // THE DECODER UNTIL *AFTER* THE NEXT CLOCK EDGE, SO ONE MORE
    // INSTRUCTION IS GOING TO ARRIVE ON THE NEXT CLOCK EDGE, AND YOU
    // BETTER HAVE THE RESOURCES TO DEAL WITH IT.
#if defined(NORENAME)
    rename_stall = false;       // inorder machine doesn't rename, so freelist never gets empty
#else
    rename_stall = (free_list_ptr < 2); // we need enough free regs to rename the next instr
#endif /* NORENAME */
  }

public:

  // inits
  rename_stage() :
    circuit(),
#if !defined(NORENAME)
    current_RAT(NUM_ARCH_REGS,0),
    retirement_RAT(NUM_ARCH_REGS,0),
    sizeof_free_list(num_phys_regs - NUM_ARCH_REGS),
    free_list(sizeof_free_list, 0),
    free_list_ptr(sizeof_free_list),
#endif

    instr_in(),
    branch_mispredict(),
    scoreboard_stall(),
    retiring_instr(),
    rob_stall(),
    
    instr_out(decoder::noop),
    rename_stall(false)
    {
#if !defined(NORENAME)
      for(uint32_t i = 0; i < NUM_ARCH_REGS; i++) {
        current_RAT[i] = i;
        retirement_RAT[i] = i;
      }
      uint32_t n = 0;
      for(uint32_t i = NUM_ARCH_REGS; i < num_phys_regs; i++) {
        free_list[n] = i;
        n++;
      }
#endif /* NORENAME */
    }
    
    // input from decoder:
    inport<decoder::instr_h> instr_in;
    
    inport<branch_packet> branch_mispredict; // indicates whether rob found a mispredict
    inport<bool> scoreboard_stall; // indicates scoreboard needs to stall fetch, decode and rename
    inport<decoder::instr_h> retiring_instr;
    inport<bool> rob_stall;
    
    // outputs
    statereg<decoder::instr_h> instr_out;
    statereg<bool> rename_stall;
};

#endif /* RENAME_STAGE_H */
