#ifndef REGFILE_CIRCUIT_H
#define REGFILE_CIRCUIT_H

#include <inttypes.h>
#include "circuit.h"

#include <utility>            // for pair<T1, T2>

using namespace std;

template <class T>
class register_file : clocked_net
{
private:
  vector<T> data_out;

  typedef pair<size_t, T> write_command;
  vector<write_command> data_in;

  void do_pulse() {
    for (size_t i = 0; i < data_in.size(); i++) {
      data_out[data_in[i].first] = data_in[i].second;
    }
    data_in.clear();
  }
public:
  friend struct lvalue : public port<const T> {
    register_file* parent;
    size_t index;
    lvalue(register_file* p, size_t i) : parent(p), index(i) { }
    void operator=(const T& new_val) {
      parent->data_in.push_back(write_command(index, new_val));
    }
    const T& operator()() const { return parent->data_out[index]; }
    void reset(const T& val) { operator=(val); parent->data_out[index] = val; }
  };

  register_file(size_t n, const T t = T()) : clocked_net(), data_out(n, t), data_in() { }

  lvalue operator[](size_t i) { return lvalue(this, i); }

  void reset() {
    // clean out everything
    for (size_t i = 0; i < data_out.size(); i++) {
      data_out[i] = T();
    }
    data_in.clear();
  }

  size_t size() { return data_out.size(); }
};

/* this specialization on register_file<bool> is necessary because
 * vector<bool> is specialized in the STL.  In particular, the bools
 * are packed in the vector, so there is no reasonable way for
 * vector<bool> operator[] to return a reference to bool.  Instead it
 * has to return a special accessor object, thus to convert
 * data_out[index] to bool& requires creating a temporary.
 * Unfortunately, our lvalue operator() needs to return such a bool&,
 * and this causes problems, because the temporary is in a scope that
 * disappears.  We solve this by specializing register_file<bool> with
 * a special version that actually uses vector<char> instead of vector<bool>. */
class register_file<bool> : clocked_net
{
private:
  vector<char> data_out;

  typedef pair<size_t, char> write_command;
  vector<write_command> data_in;

  void do_pulse() {
    for (size_t i = 0; i < data_in.size(); i++) {
      data_out[data_in[i].first] = data_in[i].second;
    }
    data_in.clear();
  }
public:
  friend struct lvalue : public port<const char> {
    register_file* parent;
    size_t index;
    lvalue(register_file* p, size_t i) : parent(p), index(i) { }
    void operator=(const bool& new_val) {
      parent->data_in.push_back(write_command(index, new_val));
    }
    const char& operator()() const { return parent->data_out[index]; }
    void reset(const bool& val) { operator=(val); parent->data_out[index] = val; }
  };

  register_file(size_t n, const bool t = false) : clocked_net(), data_out(n, t), data_in() { }

  lvalue operator[](size_t i) { return lvalue(this, i); }

  void reset() {
    // clean out everything
    for (size_t i = 0; i < data_out.size(); i++) {
      data_out[i] = false;
    }
    data_in.clear();
  }

  size_t size() { return data_out.size(); }
};

#endif /* REGFILE_CIRCUIT_H */
