#ifndef PACKETS_H_GUARD
#define PACKETS_H_GUARD

struct bus_packet {
  uint8_t reg_tag;
  uint64_t data;
  size_t sb_slot;
  size_t reorder_slot;
  bool operator==(const bus_packet& b) {
    return ((reg_tag == b.reg_tag) &&
            (data == b.data) &&
	    (sb_slot == b.sb_slot) &&
            (reorder_slot == b.reorder_slot));
  }
  bus_packet(uint8_t t = 0,
             uint64_t d = 0,
	     size_t sbs = 0,
             size_t rs = (size_t)-1) :
    reg_tag(t), data(d), sb_slot(sbs), reorder_slot(rs)
    { }
};

struct branch_packet {
  bool is_branch;
  bool taken;
  bool mispredict;
  bool is_return;
  bool is_call;
  bool local_prediction;
  bool global_prediction;
  uint32_t correct_target;
  uint32_t instr_pc;
  uint32_t global_history;
  uint32_t ra_stack_ptr;
  uint32_t ra_stack_head;
  uint32_t predicted_pc;
  size_t sb_slot;
  size_t reorder_slot;

branch_packet() : is_branch(false), taken(false), mispredict(false),
                    local_prediction(false), global_prediction(false),

                    correct_target(0),
                    instr_pc(0),
                    global_history(0),
                    sb_slot(0),
                    reorder_slot((size_t)-1)
    { }
  bool mispredicted_branch() const { return is_branch && mispredict; }

  void print() const {
    printf("branch packet: ib %d, tak %d, mis %d, lpre %d, gpre %d, ctarg: %x, pc %x, slot %d, tournament %x\n",
           is_branch, taken, mispredict, local_prediction, global_prediction, correct_target, instr_pc, reorder_slot, global_history);
  }
};

struct store_packet {
  bool is_store;
  size_t sb_slot;
  size_t reorder_slot;
  uint64_t data;

  store_packet() :
    is_store(false),
    sb_slot(0),   
    reorder_slot((size_t)-1)
    { }

  void print() const {
    printf("store packet: is %d, sb_slot %d, ro_slot %d\n", is_store, sb_slot, reorder_slot);
  }
};

struct store_buffer_packet {
  uint64_t instr_num;
  uint32_t address;
  uint64_t data;
  uint32_t size;                // 1, 2, 4 or 8 bytes
  store_buffer_packet() :
    instr_num(0xffffffffffffffffULL),
    address(0xffffffff), 
    data(0xdeadbeef),
    size(0)
    { }
};

struct event_queue_packet {
  uint64_t time;
  uint32_t vaddr;
  uint32_t caddr;

  event_queue_packet(uint64_t t=0, uint32_t v=0, uint32_t c=0) :
    time(t), vaddr(v), caddr(c) { }

  const uint64_t operator()() const { return time; }
  const bool operator==(const event_queue_packet &t) const { return time == t.time; }
  const bool operator<(const event_queue_packet &t) const { return time < t.time; }
  const bool operator<=(const event_queue_packet &t) const { return time <= t.time; }
  const bool operator>=(const event_queue_packet &t) const { return time >= t.time; }
  void print() { printf("time %llu vaddr %x", time, vaddr); }

};

struct mem_access_packet {
  bool is_access;
  bool is_store;       		// 1 if store, 0 if load
  uint32_t address;
  uint32_t size;                // 1, 2, 4 or 8 bytes
  mem_access_packet() : is_access(false), 
                        is_store(false),
                        address(0xffffffff),
                        size(0) { }
  mem_access_packet(bool i_a, bool i_s, uint32_t addr, uint32_t sz) 
    : is_access(i_a), is_store(i_s), address(addr), size(sz) { }
       
};

#endif /* PACKETS_H_GUARD */
