/****************************************************************************
 * Copyright (C) 1999 the Massachusetts Institute of Technology,
 * Cambridge, Massachusetts.  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that MIT's name not
 * be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.
 *
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * author: Matt Frank, MIT Laboratory for Computer Science,
 *         mfrank@lcs.mit.edu, 1999-Oct-19 16:19:09 EDT
 ***************************************************************************/

#ifndef __GCP_H_GUARD
#define __GCP_H_GUARD

// This is a minimal reference counted handle class.  (gcp stands for
// "garbage collected pointer").  A gcp can be declared, copied,
// compared and dereferenced, but does not support pointer arithmetic.
// When the last gcp pointing to a particular object goes out of
// scope, the object is garbage collected.
//
// CAVEATS: Reference counting has a number of problems that the user
// should keep in mind.
//
// 1. Once a pointer has been hidden in a handle you should NEVER use
// the naked pointer after that.  The handle does not know anything
// about the naked copies of the pointer that might be floating
// around, and might decide to deallocate the pointer prematurely.
// The best way to use handles is:
//    gcp<Foo> h(new Foo(blah, blah));
// Now h can be copied around arbitrarily and the Foo object will
// deallocated when the last copy of h goes out of scope.  On the
// other hand, the following would produce a heinous error:
//   Foo* fptr = new Foo(blah, blah);
//   { gcp<Foo> h(fptr); }  // h goes out of scope now
//   some_operation(*fptr)  // ACCESSING DEALLOCATED MEMORY!!! 
//
// 2. Reference counting doesn't work to garbage collect circularly
// linked structures.  This is because even though there may be no way
// to access the circular structure, each object within the structure
// still has one pointer pointing at it, with the result that the
// deallocator will not be able to collect any of the objects.
//
// 3. Reference counting has substantial overhead compared to more
// sophisticated garbage collection schemes (e.g. mark and sweep, stop
// and copy or generational collection), because the reference
// counting code is run every time a pointer is copied or deallocated.

template <class T> class gcp {
private:
  T* ptr;
  int* count;

  void decrement() {
//    fprintf(stdout, "debug: gcp     %x   decrementing count to %d\n", unsigned(ptr), (*count)-1);
    if (--(*count) <= 0) {
//      fprintf(stdout, "debug: gcp     %x   deallocating\n", unsigned(ptr));
      delete ptr;
      delete count;
    }
  }

  void increment() {
    (*count)++;
//    fprintf(stdout, "debug: gcp     %x   incrementing count to %d\n", unsigned(ptr), *count);
  }
  
public:
  typedef T element_type;
  
  explicit gcp(T* p = 0) : ptr(p), count(new int(1)) {
//    fprintf(stdout, "debug: gcp     %x initializing refcount %d\n", unsigned(ptr), *count);
  }
  
  gcp(const gcp& a) : ptr(a.ptr), count(a.count) {
//    fprintf(stdout, "debug: gcp     %x copying(constructor)\n", unsigned(ptr));
    increment();
  }

  gcp& operator=(const gcp& a) {
//    fprintf(stdout, "debug: gcp     %x copying(=)\n", unsigned(ptr));
    if (a.ptr != ptr) {
      decrement();
      ptr = a.ptr;
      count = a.count;
      increment();
    }
    return *this;
  }

  ~gcp() {
//    fprintf(stdout, "debug: gcp     %x destructor\n", unsigned(ptr));
    decrement();
  }

  T& operator*() const {
    return *ptr;
  }
  
  T* operator->() const {
    return ptr;
  }

  bool operator==(const gcp& a) const {
    return (ptr == a.ptr);
  }
  
  bool operator!=(const gcp& a) const {
    return (ptr != a.ptr);
  }
};

#endif /* __GCP_H_GUARD */
