#ifndef EXEC_UNIT_H_GUARD
#define EXEC_UNIT_H_GUARD

#include "decode.h"
#include "circuit.h"
#include "delay.h"
#include "sparse_memory.h"
#include "packets.h"
#include "bitops.h"

#include "store_buf.h"
#include "execute.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <deque>

#include "tag_file.h"

using namespace std;

// Global variables for debugging
extern uint32_t debug_mask;

// Command line global variables
extern uint32_t cache_line_psize;
extern uint32_t num_phys_regs;

class execution_unit : circuit, public state_semantics
{
  vector<uint64_t> reg_file;
  store_buffer* the_store_buffer;
  sparse_memory* the_mem;
  datapath the_datapath;

public:
  execution_unit(sparse_memory *mem) :
    circuit(),
    reg_file(num_phys_regs, 0),
    the_mem(mem),
    the_datapath(this),
    instr_in(),
    branch_mispredict(),                    // from reorder buffer
    commit_head_store(),
    writeback_bus(),
    branch_output(),
    store_output(),
    result(0),
    the_pc(0),
    the_cache(16384, 128, 4)  // 16K byte cache, 128 byte lines, 4-way set associative
    
    {
      the_store_buffer = new store_buffer(64);
    }


  // inputs - execution
  inport<decoder::instr_h> instr_in; // the instruction to execute
  inport<branch_packet> branch_mispredict; // check if a branch mispredict is retiring => cancel all the writebacks.
  inport<bool> commit_head_store;

  // outputs
  result_bus<bus_packet> writeback_bus; // register data
  statereg<branch_packet> branch_output;  // branch unit
  statereg<store_packet> store_output; // load/store unit

  uint64_t hits;
  uint64_t misses;
private:

  uint64_t result;
  uint32_t the_pc;
  decoder::instr_h the_instr;
  branch_packet branch_result;
  store_packet store_result;

  tag_file the_cache;

public:

  void recalc();

  uint64_t s_op64(decoder::instr_h instr) {
    return reg_file[instr->phys_s_reg];
  }
  uint64_t t_op64(decoder::instr_h instr) {
    return reg_file[instr->phys_t_reg];
  }
  uint64_t& dest(decoder::instr_h instr) {
    return result;
  }

  uint64_t load_int8(uint32_t vaddr) {
    return the_store_buffer->load_int8(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_uint8(uint32_t vaddr) {
    return the_store_buffer->load_uint8(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_int16(uint32_t vaddr) {
    return the_store_buffer->load_int16(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_uint16(uint32_t vaddr) {
    return the_store_buffer->load_uint16(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_int32(uint32_t vaddr) {
    return the_store_buffer->load_int32(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_uint32(uint32_t vaddr) {
    return the_store_buffer->load_uint32(the_instr->instr_num, vaddr, the_mem);
  }
  uint64_t load_uint64(uint32_t vaddr) {
    return the_store_buffer->load_uint64(the_instr->instr_num, vaddr, the_mem);
  }

  void store_uint8(uint8_t data, uint32_t vaddr) {
    the_store_buffer->insert(the_instr->instr_num, vaddr, data, 1);
  }
  void store_uint16(uint16_t data, uint32_t vaddr) {
    the_store_buffer->insert(the_instr->instr_num, vaddr, data, 2);
  }
  void store_uint32(uint32_t data, uint32_t vaddr) {
    the_store_buffer->insert(the_instr->instr_num, vaddr, data, 4);
  }
  void store_uint64(uint64_t data, uint32_t vaddr) {
    the_store_buffer->insert(the_instr->instr_num, vaddr, data, 8);
  }

  void memcpy_to_host(void* dest, const uint32_t src, size_t n) {
    the_mem->memcpy_to_host(dest, src, n);
  }
  void memcpy_from_host(uint32_t dest, const void* src, size_t n) {
    the_mem->memcpy_from_host(dest, src, n);
  }
  void strcpy_to_host(char* dest, const uint32_t src) {
    the_mem->strcpy_to_host(dest, src);
  }
  void strcpy_from_host(uint32_t dest, const char* src) {
    the_mem->strcpy_from_host(dest, src);
  }
  void emulate_syscall(uint32_t cmd_addr) {
    // noop
  }

  uint32_t& program_counter() {
    return the_pc;
  }
  void reg_init(datapath_reg_init* dri) {
    the_pc = dri->pc;
    reg_file[29] = dri->reg_29;
    reg_file[4] = dri->reg_4;
    reg_file[5] = dri->reg_5;
  }

  void halt() {
    // noop
  }

  void zero_reg_0() {
    reg_file[0] = 0;
  }

  void print_err_msg(char *s) {
    // noop (bogus instructions are often sent down pipeline speculatively)
  }
};

#endif /* EXEC_UNIT_H_GUARD */
