#include "exec_unit.h"

#include "pretty_print.h"

#include <algorithm>            // for max

extern uint64_t cycle_count;

extern uint32_t superpipeline_factor;

void
execution_unit::recalc() {

  if(debug_mask & 0x0040) {
//    printf("EXEUNT> brtopc %x brfrpc %x brmis %x isbranch %x btaken %x roslot %d wbreg %u wbdata %x wbvalid %x\n",
//           branch_output().correct_target, branch_output().instr_pc, branch_output().mispredict, 
//           branch_output().is_branch,	branch_output().taken, branch_output().reorder_slot,
//           writeback_bus().reg_tag, writeback_bus().data, writeback_bus().valid);
  }

  if (writeback_bus().reg_tag != 0) {
    if (debug_mask & 0x0400) {
      printf("\nEXEC> writeback: $%d <- %llx\n", writeback_bus().reg_tag, writeback_bus().data);
    }
    reg_file[writeback_bus().reg_tag] = writeback_bus().data;
  }

  if (commit_head_store()) {
    the_store_buffer->commit_head(the_mem);
  }

  // branch information
  branch_result.is_branch = false; 
  branch_result.taken = false;
  branch_result.mispredict = false;
  branch_result.correct_target = 0;
  branch_result.instr_pc = instr_in()->program_counter;
  branch_result.local_prediction = instr_in()->local_prediction;
  branch_result.global_prediction = instr_in()->global_prediction;
  branch_result.global_history = instr_in()->global_history;
  branch_result.ra_stack_ptr = instr_in()->ra_stack_ptr;
  branch_result.ra_stack_head = instr_in()->ra_stack_head;
  branch_result.reorder_slot = instr_in()->reorder_slot;
  branch_result.sb_slot = instr_in()->sb_slot;

  store_result.is_store = false;
  store_result.reorder_slot = instr_in()->reorder_slot;
  store_result.sb_slot = instr_in()->sb_slot;

  the_instr = instr_in();


  if (!branch_mispredict().mispredicted_branch()) {
    if (!(instr_in()->noop)) {

      result = 0;
      the_pc = instr_in()->program_counter;

      if (debug_mask & 0x0400) {
        instr_in()->print("EXEC>");
        printf("\n\t\t");
        pretty_print(instr_in());
        printf("\n");
      }

      the_datapath.execute(instr_in()); // calculate the result and valid

      // THIS IS THE CACHE LATENCY MODEL:
      if (instr_in()->is_load || instr_in()->is_store) {
        // first access is the cycle number of the first cache miss to this line:
        int64_t first_access = the_cache.access(instr_in()->vaddr, cycle_count);
        // DRAM takes 30 cycles: if we're later than first_access + 30 then the
        // line is already in the cache (0 cycle miss latency), else we have to
        // wait for the line to arrive.
        int64_t miss_latency = max((first_access+(30 * superpipeline_factor))-(int64_t)cycle_count, 0LL);

        instr_in()->bus_latency += (uint32_t)miss_latency;
        if (miss_latency > 0) {
          misses++;
        }
        else {
          hits++;
        }
      }

      if (instr_in()->phys_dest_reg != 0) { // put the result on the result bus
        size_t latency = instr_in()->bus_latency;
        bus_packet the_packet(instr_in()->phys_dest_reg,
                              result,
                              instr_in()->sb_slot,
                              instr_in()->reorder_slot);
        // arbitrate for writeback bus:
        while (writeback_bus.is_in_use(latency-1)) latency++;
        if (debug_mask & 0x0400) {
          printf("\t\t$%d will get %llx in %d\n",
                 instr_in()->phys_dest_reg,
                 result,
                 latency-1);
        }
        writeback_bus.write(the_packet, latency - 1);
      }
      if (instr_in()->is_branch || instr_in()->is_jump) {
        branch_result.is_branch = true;
        branch_result.is_call = instr_in()->is_link;
        branch_result.is_return = (instr_in()->is_indirect_jump && (instr_in()->arch_s_reg == 31));
        branch_result.correct_target = the_pc;
        branch_result.mispredict = (branch_result.correct_target != instr_in()->predicted_pc);
        branch_result.taken = (branch_result.correct_target != (instr_in()->program_counter + 4));
        branch_result.predicted_pc = instr_in()->predicted_pc;
      }
      if (instr_in()->is_store || instr_in()->is_syscall) {
        store_result.is_store = true;
      }
    }
  }
  else {
    // branch mispredict:
    writeback_bus.flush();
    the_store_buffer->clobber();
  }
  branch_output = branch_result;
  store_output = store_result;
}
