#ifndef DELAY_WRITE_H
#define DELAY_WRITE_H

#include <inttypes.h>
#include "circuit.h"
#include <vector>

using namespace std;

template <class T>
class result_bus : clocked_net, public port<const T>
{
private:
  enum { MAX_DELAY = 512 };
  vector<T> data;
  vector<bool> in_use;
  size_t time;
  T output;

  void do_pulse() {
    output = data[time];
    data[time] = T();
    in_use[time] = false;
    time = (time+1) % MAX_DELAY;
  }
public:
  result_bus() : clocked_net(), data(MAX_DELAY), in_use(MAX_DELAY), time(0) { }

  void write(const T& v, size_t delay = 0) {
    assert(delay < MAX_DELAY);
    size_t entry_num = (time + delay) % MAX_DELAY;
    assert(data[entry_num] == T()); // there should never be two people driving the bus in the same cycle
    data[entry_num] = v;
    in_use[entry_num] = true;
  }
  void flush() {
    // clean out everything in the fifo:
    for (size_t i = 0; i < data.size(); i++) {
      data[i] = T();
      in_use[i] = false;
    }
  }
  bool is_in_use(size_t delay = 0) {
    size_t entry_num = (time + delay) % MAX_DELAY;
    return in_use[entry_num];
  }
  const T& operator()() const { return output; }
};

#endif /* DELAY_WRITE_H */
