#ifndef ARCH_MODEL_H_GUARD
#define ARCH_MODEL_H_GUARD

#include <vector>
#include "decode.h"
#include "execute.h"
#include "sparse_memory.h"

class arch_model : public state_semantics
{
private:
  vector< uint64_t > reg_file;
  uint32_t the_pc;
  sparse_memory* the_mem;
  datapath the_datapath;
public:
  arch_model(sparse_memory* mem) :
    state_semantics(),
    reg_file(32, 0),
    the_pc(99),
    the_mem(mem),
    the_datapath(this),
    halted(false)
    { }

  bool halted;

  uint64_t s_op64(decoder::instr_h instr) { return reg_file[instr->arch_s_reg]; }
  uint64_t t_op64(decoder::instr_h instr) { return reg_file[instr->arch_t_reg]; }
  uint64_t& dest(decoder::instr_h instr)  { return reg_file[instr->arch_dest_reg]; }
  uint64_t load_int8(uint32_t vaddr) { return the_mem->load_int8(vaddr); }
  uint64_t load_uint8(uint32_t vaddr) { return the_mem->load_uint8(vaddr); }
  uint64_t load_int16(uint32_t vaddr) { return the_mem->load_int16(vaddr); }
  uint64_t load_uint16(uint32_t vaddr) { return the_mem->load_uint16(vaddr); }
  uint64_t load_int32(uint32_t vaddr) { return the_mem->load_int32(vaddr); }
  uint64_t load_uint32(uint32_t vaddr) { return the_mem->load_uint32(vaddr); }
  uint64_t load_uint64(uint32_t vaddr) { return the_mem->load_uint64(vaddr); }
  void store_uint8(uint8_t data, uint32_t vaddr) { the_mem->store_uint8(data, vaddr); }
  void store_uint16(uint16_t data, uint32_t vaddr) { the_mem->store_uint16(data, vaddr); }
  void store_uint32(uint32_t data, uint32_t vaddr) { the_mem->store_uint32(data, vaddr); }
  void store_uint64(uint64_t data, uint32_t vaddr) { the_mem->store_uint64(data, vaddr); }
  void memcpy_to_host(void* dest, const uint32_t src, size_t n) { the_mem->memcpy_to_host(dest, src, n); }
  void memcpy_from_host(uint32_t dest, const void* src, size_t n) { the_mem->memcpy_from_host(dest, src, n); }
  void strcpy_to_host(char* dest, const uint32_t src) { the_mem->strcpy_to_host(dest, src); }
  void strcpy_from_host(uint32_t dest, const char* src) { the_mem->strcpy_from_host(dest, src); }
  void emulate_syscall(uint32_t cmd_addr) { the_mem->emulate_syscall(cmd_addr); }
  uint32_t& program_counter() { return the_pc; }
  void zero_reg_0() { reg_file[0] = 0; }
  void print_err_msg(char *s) { fprintf(stderr, "%s", s); }
  decoder::instr_h do_cycle() {
    decoder::instr_h the_instr = decoder::run(the_mem->load_uint32(the_pc));

    the_instr->program_counter = the_pc;

    the_datapath.execute(the_instr);
    return the_instr;
  }

  void halt() { halted = true; }

  int load_elf_file(const char* filename, int argc, char* argv[], datapath_reg_init* dri) {
    return the_datapath.load_elf_file(filename, argc, argv, dri);
  }

  void init_regs(datapath_reg_init* dri) {
    the_pc = dri->pc;
    reg_file[29] = dri->reg_29;
    reg_file[4] = dri->reg_4;
    reg_file[5] = dri->reg_5;
  }

  ~arch_model() { }
};

#endif /* ARCH_MODEL_H_GUARD */
