(* File: common.ml *)
(* This version has app_exn_handler_cont in it! *)

(* expressions for PicoML *)
type const = BoolConst of bool | IntConst of int | FloatConst of float
           | StringConst of string | NilConst | UnitConst 

let string_of_const c =
    match c 
    with IntConst n    -> if n < 0 then "~"^string_of_int(abs n) else string_of_int n
       | BoolConst b   -> if b then "true" else "false"
       | FloatConst f  -> ((string_of_float f)^(if ceil f = floor f then ("0") else ("")))
       | StringConst s -> ("\""^ (String.escaped s)^ "\"")
       | NilConst      -> "[]"
       | UnitConst     -> "()"

 type mon_op = IntNegOp | HdOp | TlOp | FstOp | SndOp | PrintStringOp

let string_of_mon_op m =
    match m
    with IntNegOp   -> "~"
       | HdOp  -> "hd"
       | TlOp  -> "tl"
       | FstOp   -> "fst"
       | SndOp   -> "snd"
       | PrintStringOp  -> "print_string"

type bin_op = IntPlusOp | IntMinusOp | IntTimesOp | IntDivOp
           | FloatPlusOp | FloatMinusOp | FloatTimesOp | FloatDivOp 
           | ConcatOp | ConsOp | CommaOp | EqOp | GreaterOp 
           | ModOp | ExpoOp

let string_of_bin_op = function 
     IntPlusOp  -> " + "
   | IntMinusOp -> " - "
   | IntTimesOp -> " * "
   | IntDivOp -> " / "
   | FloatPlusOp -> " +. "
   | FloatMinusOp -> " -. "
   | FloatTimesOp -> " *. "
   | FloatDivOp -> " /. "
   | ConcatOp -> " ^ "
   | ConsOp -> " :: "
   | CommaOp -> " , "
   | EqOp  -> " = "
   | GreaterOp -> " > "
   | ExpoOp -> "**"
   | ModOp   -> "mod"

type exp =  (* Exceptions will be added in later MPs *)
   | VarExp of string                    (* variables *)
   | ConstExp of const                   (* constants *)
   | MonOpAppExp of mon_op * exp         (* % e1 for % is a builtin monadic operator *) 
   | BinOpAppExp of bin_op * exp * exp   (* e1 % e2 for % is a builtin binary operator *)
   | IfExp of exp * exp * exp            (* if e1 then e2 else e3 *)
   | AppExp of exp * exp                 (* e1 e2 *) 
   | FunExp of string * exp              (* fun x -> e1 *)
   | LetInExp of string * exp * exp      (* let x = e1 in e2 *)
   | LetRecInExp of string * string * exp * exp (* let rec f x = e1 in e2 *)
   | RaiseExp of exp                     (* raise e *)
   | TryWithExp of (exp * int option * exp * (int option * exp) list)
		                   (* try e with i -> e1 | j -> e1 | ... | k -> en *)

type dec =
     Anon of exp
   | Let of string * exp                 (* let x = exp *)
   | LetRec of string * string * exp     (* let rec f x = exp *)

let rec string_of_exp = function
   VarExp s -> s
 | ConstExp c ->  string_of_const c
 | IfExp(e1,e2,e3)->"if " ^ (string_of_exp e1) ^
                 " then " ^ (string_of_exp e2) ^
                 " else " ^ (string_of_exp e3)
 | MonOpAppExp (m,e) ->  (string_of_mon_op m) ^ " " ^ (paren_string_of_exp e) 
 | BinOpAppExp (b,e1,e2) -> 
   (match b with CommaOp -> ("(" ^ (paren_string_of_exp e1) ^ (string_of_bin_op b) ^
                              (paren_string_of_exp e2) ^ ")")
    | _ -> ((paren_string_of_exp e1) ^ " " ^ (string_of_bin_op b)
            ^ " " ^ (paren_string_of_exp e2)))
 | AppExp(e1,e2) -> (non_app_paren_string_of_exp e1) ^ " " ^ (paren_string_of_exp e2) 
 | FunExp (x,e) ->  ("fun " ^ x ^ " -> " ^ (string_of_exp e))
 | LetInExp (x,e1,e2) -> ("let "^x^" = "^ (string_of_exp e1) ^ " in " ^ (string_of_exp e2))
 | LetRecInExp (f,x,e1,e2) -> 
    ("let rec "^f^" "^x^" = "^(string_of_exp e1) ^ " in " ^ (string_of_exp e2))
 | RaiseExp e -> "raise " ^ (string_of_exp e)
 | TryWithExp (e,intopt1,exp1,match_list) ->
    "try " ^ (paren_string_of_exp e) ^  " with " ^
     (string_of_exc_match (intopt1,exp1)) ^
     (List.fold_left (fun s m -> (s^" | " ^ (string_of_exc_match m))) "" match_list) 

and paren_string_of_exp e =
    match e with VarExp _ | ConstExp _ -> string_of_exp e
    | _ -> "(" ^ string_of_exp e ^ ")"

and non_app_paren_string_of_exp e =
    match e with AppExp (_,_) -> string_of_exp e
    | _ -> paren_string_of_exp e
 

and string_of_exc_match (int_opt, e) =
    (match int_opt with None -> "_" | Some n -> string_of_int n) ^
    " -> " ^
    (string_of_exp e)
							
let string_of_dec = function
 | Anon e -> ("let _ = "^ (string_of_exp e))
 | Let (s, e) ->  ("let "^ s ^" = " ^ (string_of_exp e))
 | LetRec (fname,argname,fn) -> 
    ("let rec " ^ fname ^ " " ^ argname ^ " = " ^ (string_of_exp fn))

let print_exp exp = print_string (string_of_exp exp) 
let print_dec dec = print_string (string_of_dec dec)

(* Util functions *)
let rec drop y = function
   []    -> []
 | x::xs -> if x=y then drop y xs else x::drop y xs

let rec delete_duplicates = function
   []    -> []
 | x::xs -> x::delete_duplicates (drop x xs)

(********************** Type System Infrastructure **********************)

type typeVar = int

let rec expand n (list,len) =
    let q = n / 26 in
        if q = 0 then (n :: list, len + 1)
        else expand q (((n mod 26)::list), len + 1);;

let string_of_typeVar n = 
   let (num_list,len) =
       match (expand n ([],0))
       with ([],l) -> ([],l) (* can't actually happen *)
          | ([s],l) -> ([s],l)
          | (x::xs,l) -> ((x - 1) :: xs, l)
   in
   let s = (Bytes.create len)
   in
   let _ =
    List.fold_left
    (fun n c -> (Bytes.set s n c; n + 1))
    0
    (List.map (fun x -> Char.chr(x + 97)) num_list)  (* Char.code 'a' = 97 *)
   in "'"^(Bytes.to_string s);;

type monoTy = TyVar of typeVar | TyConst of (string * monoTy list)

let rec string_of_monoTy t =
  let rec string_of_tylist = function
     []     -> ""
   | t'::[] -> string_of_monoTy t'
   | t'::ts -> string_of_monoTy t'^ ","^ string_of_tylist ts
  in
  let string_of_subty s =
  match s with 
     TyConst ("*", _) | TyConst ("->", _) -> ("("^ string_of_monoTy s^ ")")
   | _ ->  string_of_monoTy s
  in 
    match t with
       TyVar n         -> (string_of_typeVar n)
     |TyConst (name, []) -> name
     |TyConst (name, [ty]) -> (string_of_subty ty^ " "^ name)
     |TyConst ("*", [ty1; ty2]) -> (string_of_subty ty1^ " * "^ string_of_monoTy ty2)
     |TyConst ("->", [ty1; ty2]) -> (string_of_subty ty1^ " -> "^ string_of_monoTy ty2)
     |TyConst (name, tys) -> ("("^ string_of_tylist tys^ ") "^ name)

let rec accummulate_freeVarsMonoTy fvs ty =
    match ty
    with TyVar n -> n::fvs
       | TyConst (c, tyl) -> List.fold_left accummulate_freeVarsMonoTy fvs tyl

let freeVarsMonoTy ty = delete_duplicates (accummulate_freeVarsMonoTy [] ty)

(*fresh type variable*)
let (fresh, reset) =
   let nxt = ref 0 in
   let f () = (nxt := !nxt + 1; TyVar(!nxt)) in
   let r () = nxt := 0 in
    (f, r)

let bool_ty = TyConst("bool",[])
let int_ty = TyConst ("int", [])
let float_ty = TyConst ("float",[])
let string_ty = TyConst ("string",[])
let unit_ty = TyConst("unit", [])
let mk_pair_ty ty1 ty2 = TyConst("*",[ty1;ty2])
let mk_fun_ty ty1 ty2 = TyConst("->",[ty1;ty2])
let mk_list_ty ty = TyConst("list",[ty])

type polyTy = typeVar list * monoTy  (* the list is for quantified variables *)

let string_of_polyTy (bndVars, t) = match bndVars with [] -> string_of_monoTy t
    | _ ->  (List.fold_left
             (fun s v -> s ^ " " ^ string_of_typeVar v)
             "Forall"
             bndVars)
             ^ ". " ^ string_of_monoTy t

let freeVarsPolyTy ((tvs, ty):polyTy) = delete_duplicates(
    List.filter (fun x -> not(List.mem x tvs)) (freeVarsMonoTy ty))

let polyTy_of_monoTy mty = (([],mty):polyTy)

let int_op_ty = polyTy_of_monoTy(mk_fun_ty int_ty (mk_fun_ty int_ty int_ty))
let float_op_ty =
    polyTy_of_monoTy(mk_fun_ty float_ty (mk_fun_ty float_ty float_ty))
let string_op_ty =
    polyTy_of_monoTy(mk_fun_ty string_ty (mk_fun_ty string_ty string_ty))

(* fixed signatures *)
let const_signature const = match const with
   BoolConst b -> (([], bool_ty):polyTy)
 | IntConst n -> ([], int_ty)
 | FloatConst f -> ([], float_ty)
 | StringConst s -> ([], string_ty)
 | NilConst -> ([0],mk_list_ty (TyVar 0))
 | UnitConst -> ([], unit_ty)

let binop_signature binop = match binop with
     IntPlusOp   -> int_op_ty
   | IntMinusOp   -> int_op_ty
   | IntTimesOp   -> int_op_ty
   | IntDivOp   -> int_op_ty
   | ModOp      -> int_op_ty
   | ExpoOp     -> float_op_ty
   | FloatPlusOp   -> float_op_ty
   | FloatMinusOp   -> float_op_ty
   | FloatTimesOp   -> float_op_ty
   | FloatDivOp   -> float_op_ty
   | ConcatOp -> string_op_ty
   | ConsOp -> 
       let alpha = TyVar 0
       in ([0], 
              mk_fun_ty alpha (mk_fun_ty (mk_list_ty alpha) (mk_list_ty alpha)))
   | CommaOp ->
       let alpha = TyVar 0 in
       let beta = TyVar 1 in
           ([0;1],
            mk_fun_ty alpha (mk_fun_ty beta (mk_pair_ty alpha beta)))
(*   | EqOp -> ([],mk_fun_ty int_ty (mk_fun_ty int_ty bool_ty)) *)
   | EqOp ->
     let alpha = TyVar 0 in ([0],mk_fun_ty alpha (mk_fun_ty alpha bool_ty))
   | GreaterOp ->
     let alpha = TyVar 0 in ([0],mk_fun_ty alpha (mk_fun_ty alpha bool_ty))

let monop_signature monop = match monop with
    | HdOp -> let alpha = TyVar 0 in([0], mk_fun_ty (mk_list_ty alpha) alpha)
    | TlOp -> let alpha = TyVar 0 in
                  ([0], mk_fun_ty (mk_list_ty alpha) (mk_list_ty alpha))
    | PrintStringOp -> ([], mk_fun_ty string_ty unit_ty)
    | IntNegOp -> ([], mk_fun_ty int_ty int_ty)
    | FstOp -> let t1,t2 = TyVar 0,TyVar 1
             in ([0;1],mk_fun_ty (mk_pair_ty t1 t2) t1)
    | SndOp -> let t1,t2 = TyVar 0,TyVar 1
             in ([0;1],mk_fun_ty (mk_pair_ty t1 t2) t2)

(* environments *)
type 'a env = (string * 'a) list

let freeVarsEnv l = delete_duplicates (
    List.fold_right (fun (_,pty) fvs -> freeVarsPolyTy pty @ fvs) l [])

let string_of_env string_of_entry gamma = 
  let rec string_of_env_aux gamma =
    match gamma with
       []        -> ""
     | (x,y)::xs -> x^ " : "^ string_of_entry y^
                    match xs with [] -> "" | _  -> ", "^
                                                   string_of_env_aux xs
  in
    "{"^ string_of_env_aux gamma^ "}"

let string_of_type_env gamma = string_of_env string_of_polyTy gamma

(*environment operations*)
let rec lookup mapping x =
  match mapping with
     []        -> None
   | (y,z)::ys -> if x = y then Some z else lookup ys x

type type_env = polyTy env

let make_env x y = ([(x,y)]:'a env)
let lookup_env (gamma:'a env) x = lookup gamma x
let sum_env (delta:'a env) (gamma:'a env) = ((delta@gamma):'a env)
let ins_env (gamma:'a env) x y = sum_env (make_env x y) gamma

(****************** Unification, Adapted from MP4 ***********************)

type substitution = (typeVar * monoTy) list

let subst_fun (s:substitution) n = (try List.assoc n s with _ -> TyVar n)

(*unification algorithm*)
(* Problem 1 *)
let rec contains n ty =
  match ty with
    TyVar m -> n=m
  | TyConst(st, typelst) ->
     List.fold_left (fun xl x -> if xl then xl else contains n x) false typelst;;

(* Problem 2 *)
let rec substitute ie ty = 
  let n,sub = ie 
  in match ty with
       TyVar m -> if n=m then sub else ty
     | TyConst(st, typelist) -> TyConst(st, List.map (fun t -> substitute ie t) typelist);;

let polyTySubstitute s (pty:polyTy) =
    match s with  (n,residue) ->
    (match pty with (bound_vars, ty) -> 
           if List.mem n bound_vars then pty
           else ((bound_vars, substitute s ty):polyTy))
    

(* Problem 3 *)
let rec monoTy_lift_subst (s:substitution) ty =
  match ty with
    TyVar m -> subst_fun s m
  | TyConst(st, typelst) ->  TyConst(st, List.map (fun t -> monoTy_lift_subst s t) typelst);;

(* Problem 4 *)
let rec unify eqlst : substitution option =
  let rec addNewEqs lst1 lst2 acc =
    match lst1,lst2 with
      [],[] -> Some acc
    | t::tl, t'::tl' -> addNewEqs tl tl' ((t,t')::acc)
    | _ -> None
  in
  match eqlst with
    [] -> Some([])
    (* Delete *)
  | (s,t)::eqs when s=t -> unify eqs
    (* Eliminate *)
  | (TyVar(n),t)::eqs when not(contains n t)-> 
      let eqs' = List.map (fun (t1,t2) -> (substitute (n,t) t1 , substitute (n,t) t2)) eqs
      in (match unify eqs' with
           None -> None
         | Some(phi) -> Some((n, monoTy_lift_subst phi t):: phi))
    (* Orient *)
  | (TyConst(str, tl), TyVar(m))::eqs -> unify ((TyVar(m), TyConst(str, tl))::eqs)
    (* Decompose *)
  | (TyConst(str, tl), TyConst(str', tl'))::eqs when str=str' -> 
      (match (addNewEqs tl tl' eqs) with
        None -> None
      | Some l -> unify l)
    (* Other *)
  | _ -> None
;;


(****************** Support Infrastructure for  ML3 ***********************)

(*judgment*) 
type judgment =
   ExpJudgment of type_env * exp * monoTy
 | DecJudgment of type_env * dec * type_env

let string_of_judgment judgment =
  match judgment with ExpJudgment(gamma, exp, monoTy) ->
        string_of_type_env gamma ^ " |= "^ string_of_exp exp ^
         " : " ^ string_of_monoTy monoTy
  | DecJudgment (gamma, dec, delta) ->
        string_of_type_env gamma ^ " |= "^ string_of_dec dec ^
         " : " ^ string_of_type_env delta

type proof = Proof of proof list * judgment

(*proof printing*)
let string_of_proof p =
  let depth_max = 10 in
  let rec string_of_struts = function
     []    -> ""
   | x::[] -> (if x then "|-" else "|-")  (* ??? *)
   | x::xs -> (if x then "  " else "| ")^ string_of_struts xs
  in let rec string_of_proof_aux (Proof(ant,conc)) depth lst =
    "\n"^ "  "^ string_of_struts lst^
    (if (depth > 0) then "-" else "")^
    let assum = ant in
      string_of_judgment conc ^
      if depth <= depth_max
         then string_of_assum depth lst assum
      else ""
  and string_of_assum depth lst assum =
    match assum with 
       []     -> ""
     | p'::ps -> string_of_proof_aux p' (depth + 1) (lst@[ps=[]])^
                 string_of_assum depth lst ps
  in
    string_of_proof_aux p 0 []^ "\n\n"

let rec monoTy_rename_tyvars s mty =
    match mty with
      TyVar n -> (match lookup s n with Some m -> TyVar m | _ -> mty)
    | TyConst(c, tys) -> TyConst(c, List.map (monoTy_rename_tyvars s) tys)

let subst_compose (s2:substitution) (s1:substitution) : substitution =
    (List.filter (fun (tv,_) -> not(List.mem_assoc tv s1)) s2) @ 
    (List.map (fun (tv,residue) -> (tv, monoTy_lift_subst s2 residue)) s1)

let gen (env:type_env) ty =
    let env_fvs = freeVarsEnv env in
    ((List.filter (fun v -> not(List.mem v env_fvs)) (freeVarsMonoTy ty), ty):polyTy)

let freshInstance ((tvs, ty):polyTy) =
    let fresh_subst = List.fold_right (fun tv s -> ((tv,fresh())::s)) tvs [] in
    monoTy_lift_subst fresh_subst ty

let first_not_in n l =
    let rec first m n l =
        if n > 0 then
         if List.mem m l then first (m+1) n l else m :: (first (m+1) (n - 1) l)
        else []
    in first 0 n l

let alpha_conv ftvs (pty:polyTy) =
    match pty with (btvs, ty) ->
    (let fresh_bvars =
         first_not_in (List.length btvs) (ftvs @ (freeVarsPolyTy pty))
     in (fresh_bvars,
         monoTy_lift_subst (List.combine btvs (List.map (fun v -> TyVar v) fresh_bvars))
         ty))

let polyTy_lift_subst s pty =
	let rec fvsfun x r = match x with
		| TyVar n -> n :: r
		| TyConst (_, l) -> List.fold_right fvsfun l r
	in
	let fvs = List.fold_right fvsfun (snd(List.split s)) [] in
    let (nbvs, nty) = alpha_conv fvs pty in
    ((nbvs, monoTy_lift_subst s nty):polyTy)

let rec mk_bty_renaming n bty =
    match bty with [] -> ([],[])
    | (x::xs) -> (match mk_bty_renaming (n-1) xs
                   with (s,l) -> (((x,n) :: s), n :: l))

let polyTy_rename_tyvars s (bty, mty) =
    let (renaming,new_bty) = mk_bty_renaming (~-7) bty in
    (new_bty, monoTy_rename_tyvars s (monoTy_rename_tyvars renaming mty))

let env_rename_tyvars s (env: 'a env) =
    ((List.map
      (fun (x,polyTy) -> (x,polyTy_rename_tyvars s polyTy)) env): 'a env)

let env_lift_subst s (env:'a env) =
    ((List.map (fun (x,polyTy) -> (x,polyTy_lift_subst s polyTy)) env):'a env)


(*constraint list*)
type consList = (monoTy * monoTy) list


(*applying a substitution to a proof*)

(*applying a substitution to a proof*)
let rec proof_lift_subst f = function
    Proof(assum, ExpJudgment(gamma, exp, monoTy)) ->
    Proof(List.map (proof_lift_subst f) assum,
          ExpJudgment(env_lift_subst f gamma, exp, monoTy_lift_subst f monoTy))
 | Proof(assum, DecJudgment(gamma, dec, delta)) ->
    Proof(List.map (proof_lift_subst f) assum,
          DecJudgment(env_lift_subst f gamma, dec, env_lift_subst f delta))

let rec proof_rename_tyvars f = function
    Proof(assum, ExpJudgment(gamma, exp, monoTy)) ->
    Proof(List.map (proof_rename_tyvars f) assum,
          ExpJudgment(env_rename_tyvars f gamma, exp,
                      monoTy_rename_tyvars f monoTy))
 | Proof(assum, DecJudgment(gamma, dec, delta)) ->
    Proof(List.map (proof_rename_tyvars f) assum,
          DecJudgment(env_rename_tyvars f gamma, dec,
                      env_rename_tyvars f delta))

let get_ty = function
   None       -> raise(Failure "None")
 | Some(ty,p) -> ty

let get_proof = function
   None       -> raise(Failure "None")
 | Some(ty,p) -> p

let infer_exp gather_exp (gamma:type_env) (exp:exp) = 
  let ty = fresh() in
  let result = 
    match gather_exp gamma exp ty with
       None         -> None
     | Some(proof,sigma) -> match ty with
          | TyVar n -> Some (subst_fun sigma n, proof_lift_subst sigma proof)
          | _       -> None
  in let _ = reset() in
  result;;

let infer_dec gather_dec (gamma:type_env) (dec:dec) =
  let result = 
    match gather_dec gamma dec with
       None -> None
     | Some(proof,sigma) -> Some (proof_lift_subst sigma proof)
  in let _ = reset() in
  result;;

let string_of_constraints c =
  let rec aux c =
     match c with 
     | [] -> ""
     | [(s,t)] ->  (string_of_monoTy s^ " --> "^ string_of_monoTy t)
     | (s,t)::c' -> (string_of_monoTy s^ " --> "^ string_of_monoTy t^
		     "; "^ aux c')
  in ("["^ aux c^ "]\n")

 
let string_of_substitution s =
  let rec aux s =
     match s with 
     | [] -> ""
     | [(i,t)] -> ((string_of_typeVar i)  ^ " |--> " ^ string_of_monoTy t)
     | (i,t)::s' -> (((string_of_typeVar i)  ^ " |--> ")^
                     string_of_monoTy t^ "; "^ aux s')
  in ("["^ aux s^ "]\n")


let niceInfer_exp gather_exp (gamma:type_env) exp = 
  let ty = fresh()
  in
  let result = 
    match gather_exp gamma exp ty with
     None ->
      (print_string("Failure: No type for expression: "^
       string_of_exp exp^ "\n"^
       "in the environment: "^
       string_of_env string_of_polyTy gamma^ "\n");
       raise (Failure ""))
   | Some (p,s) ->
   (string_of_proof p^
	(*
   "Constraints: "^
   string_of_constraints c ^
   "Unifying..."^
   match unify c with
     None -> ("Failure: No solution for these constraints!\n"^
              raise (Failure ""))
   | Some s ->
	*)
   ("Unifying substitution: "^
    string_of_substitution s^
    "Substituting...\n"^
    let new_p = proof_lift_subst s p in
    string_of_proof new_p)) in
  let _ = reset() in
  result;;

let niceInfer_dec
    (gather_dec:(type_env -> dec -> (proof * type_env * substitution) option))
    (gamma:type_env) dec = 
  let result = 
    match gather_dec gamma dec with
     None ->
      (print_string("Failure: No type for declaraion: "^
       string_of_dec dec^ "\n"^
       "in the environment: "^
       string_of_env string_of_polyTy gamma^ "\n");
       raise (Failure ""))
   | Some (p,d,s) ->
   (string_of_proof p^
   ("Unifying substitution: "^
    string_of_substitution s^
    "Substituting...\n"^
    let new_p = proof_lift_subst s p in
    string_of_proof new_p)) in
  let _ = reset() in
  result;;

(* Collect all the TyVar indices in a proof *)

let rec collectTypeVars ty lst =
  match ty with
    TyVar m -> m::lst
  | TyConst(st, typelst) -> List.fold_left (fun xl x -> collectTypeVars x xl) lst typelst

let rec collectFreeTypeVars bty ty lst =
  match ty with
    TyVar m -> if List.mem m bty then lst else m::lst
  | TyConst(st, typelst) ->
    List.fold_left (fun xl x -> collectFreeTypeVars bty x xl) lst typelst

let collectPolyTyVars (bty,mty) lst = collectFreeTypeVars bty mty lst

let collectEnvVars (gamma:type_env) lst =
    List.fold_left (fun tys (_,pty)-> collectPolyTyVars pty tys) lst gamma

let collectJdgVars jdg lst =
    match jdg with ExpJudgment(gamma, exp, monoTy) ->
        collectEnvVars gamma (collectTypeVars monoTy lst)
    | DecJudgment(gamma, dec, delta) ->
        collectEnvVars gamma (collectEnvVars delta lst)

let rec collectProofVars prf lst =
  match prf with Proof (assum, jdg)
   -> collectAssumVars assum (collectJdgVars jdg lst)
and collectAssumVars assum lst =
  match assum with 
    []     -> lst
  | p::ps -> collectAssumVars ps (collectProofVars p lst)

let canonicalize_proof prf_opt =
    match prf_opt with None -> None
    | Some(ty, prf) ->
  let (varlst,_) =
    List.fold_right (fun x (xl,idx) -> ((x,idx)::xl), idx+1) 
      (delete_duplicates (collectProofVars prf (collectTypeVars ty []))) 
      ([],1)
  in Some(monoTy_rename_tyvars varlst ty, proof_rename_tyvars varlst prf)

let canon = canonicalize_proof

let canon_dec prf_opt =
    match prf_opt with None -> None
    | Some prf ->
  let (varlst,_) =
    List.fold_right (fun x (xl,idx) -> ((x, idx)::xl), idx+1) 
      (delete_duplicates (collectProofVars prf []))
      ([],1)
  in Some(proof_rename_tyvars varlst prf)

(********************* MP6's inferencer *********************)

let rec gather_exp_ty_substitution gamma exp tau =
    let judgment = ExpJudgment(gamma, exp, tau) in
    match exp
    with ConstExp c ->
         let tau' = const_signature c in
         (match unify [(tau, freshInstance tau')]
          with None       -> None
             | Some sigma -> Some(Proof([],judgment), sigma))
    | VarExp x -> 
      (match lookup_env gamma x with None -> None
       | Some gamma_x ->
         (match unify [(tau, freshInstance gamma_x)]
          with None       -> None
             | Some sigma -> Some(Proof([],judgment), sigma)))
    | BinOpAppExp (binop, e1,e2) ->
      let tau' = binop_signature binop in
      let tau1 = fresh() in
      let tau2 = fresh() in
      (match gather_exp_ty_substitution gamma e1 tau1
       with None -> None
       | Some(pf1, sigma1) ->
         (match gather_exp_ty_substitution (env_lift_subst sigma1 gamma) e2 tau2
          with None -> None
          | Some (pf2, sigma2) ->
            let sigma21 = subst_compose sigma2 sigma1 in
            (match unify[(monoTy_lift_subst sigma21
                          (mk_fun_ty tau1 (mk_fun_ty tau2 tau)),
                         freshInstance tau')]
             with None -> None
             | Some sigma3 -> 
               Some(Proof([pf1;pf2], judgment),subst_compose sigma3 sigma21))))
    | MonOpAppExp (monop, e1) ->
      let tau' = monop_signature monop in
      let tau1 = fresh() in
      (match gather_exp_ty_substitution gamma e1 tau1
       with None -> None
       | Some(pf, sigma) ->
         (match unify[(monoTy_lift_subst sigma (mk_fun_ty tau1 tau),
                       freshInstance tau')]
          with None -> None
          | Some subst ->
            Some(Proof([pf], judgment),
                 subst_compose subst sigma)))
    | IfExp(e1,e2,e3) ->
      (match gather_exp_ty_substitution gamma e1 bool_ty
       with None -> None
       | Some(pf1, sigma1) ->
         (match gather_exp_ty_substitution
                (env_lift_subst sigma1 gamma) e2 (monoTy_lift_subst sigma1 tau)
          with None -> None
          | Some (pf2, sigma2) ->
            let sigma21 = subst_compose sigma2 sigma1 in
            (match gather_exp_ty_substitution
                   (env_lift_subst sigma21 gamma) e3
                   (monoTy_lift_subst sigma21 tau)
             with  None -> None
             | Some(pf3, sigma3) ->
               Some(Proof([pf1;pf2;pf3], judgment), subst_compose sigma3 sigma21))))
    | FunExp(x,e) ->
      let tau1 = fresh() in
      let tau2 = fresh() in
      (match gather_exp_ty_substitution
             (ins_env gamma x (polyTy_of_monoTy tau1)) e tau2
       with None -> None
       | Some (pf, sigma) ->
         (match unify [(monoTy_lift_subst sigma tau,
                        monoTy_lift_subst sigma (mk_fun_ty tau1 tau2))]
          with None -> None
          | Some sigma1 ->
            Some(Proof([pf],judgment), subst_compose sigma1 sigma)))
    | AppExp(e1,e2) ->
      let tau1 = fresh() in
      (match gather_exp_ty_substitution gamma e1 (mk_fun_ty tau1 tau)
       with None -> None
       | Some(pf1, sigma1) ->
         (match gather_exp_ty_substitution (env_lift_subst sigma1 gamma) e2
                                           (monoTy_lift_subst sigma1 tau1)
          with None -> None
          | Some (pf2, sigma2) ->
            Some(Proof([pf1;pf2], judgment), subst_compose sigma2 sigma1)))
    | RaiseExp e ->
      (match gather_exp_ty_substitution gamma e int_ty
       with None -> None
       | Some(pf, sigma) -> Some(Proof([pf],judgment), sigma))
    | LetInExp(x,e1,e2)  -> 
       let tau1 = fresh() in
       (match gather_exp_ty_substitution gamma e1 tau1
	with None -> None
	   | Some(pf1, sigma1) -> 
	      let delta_env = make_env x (gen (env_lift_subst sigma1 gamma) 
					      (monoTy_lift_subst sigma1 tau1)) in
	      (match gather_exp_ty_substitution 
		       (sum_env delta_env (env_lift_subst sigma1 gamma)) e2
                         (monoTy_lift_subst sigma1 tau)
	       with None -> None
		  | Some (pf2,sigma2) ->
		     let sigma21 = subst_compose sigma2 sigma1 in
		     Some(Proof([pf1;pf2], judgment), sigma21)))
    | LetRecInExp(f,x,e1,e2) ->
       let tau1  = fresh() in
       let tau2 = fresh() in
       let tau1_to_tau2 = mk_fun_ty tau1 tau2 in
       (match gather_exp_ty_substitution
		(ins_env (ins_env gamma f (polyTy_of_monoTy tau1_to_tau2))
			  x (polyTy_of_monoTy tau1))
		e1 tau2
	with None -> None
	   | Some(pf1, sigma1) -> 
              let sigma1_gamma = env_lift_subst sigma1 gamma in
	      let sigma1_tau1_to_tau2 = monoTy_lift_subst sigma1 tau1_to_tau2 in
	      (match gather_exp_ty_substitution
                     (ins_env sigma1_gamma f (gen sigma1_gamma sigma1_tau1_to_tau2))
		     e2 (monoTy_lift_subst sigma1 tau)
	       with None -> None
		  | Some(pf2,sigma2) ->
		     let sigma21 = subst_compose sigma2 sigma1 in
		     Some(Proof([pf1;pf2], judgment),sigma21)))
    | TryWithExp (e,intopt1,e1, match_list) ->
      (match (gather_exp_ty_substitution gamma e tau)
       with None -> None
       | Some (pf, sigma) ->
         (match
           List.fold_left
           (fun part_result -> fun (intopti, ei) ->
            (match part_result with None -> None
             | Some (rev_pflist, comp_sigmas) ->
               (match gather_exp_ty_substitution
                      (env_lift_subst comp_sigmas gamma) ei
                      (monoTy_lift_subst comp_sigmas tau)
                with None -> None
                | Some (pfi, sigmai) ->
                  Some (pfi :: rev_pflist, subst_compose sigmai comp_sigmas))))
           (Some([pf], sigma))
           ((intopt1,e1):: match_list)
           with None -> None
           | Some (rev_pflist, comp_subst) ->
             Some(Proof(List.rev rev_pflist, judgment), comp_subst)))


let rec gather_dec_ty_substitution gamma dec =
    match dec with 
    | Anon e ->
      let tau = fresh() in
      (match gather_exp_ty_substitution gamma e tau
	with None -> None
	   | Some(pf, sigma) ->
             Some(Proof([pf],DecJudgment (gamma, dec, [])), sigma))
    | Let(x,e) -> 
       let tau = fresh() in
       (match gather_exp_ty_substitution gamma e tau
	with None -> None
	   | Some(pf, sigma) -> 
	      let delta_env = make_env x (gen (env_lift_subst sigma gamma) 
					      (monoTy_lift_subst sigma tau)) in
             Some(Proof([pf],DecJudgment (gamma, dec, delta_env)),sigma))
    | LetRec(f,x,e) ->
       let tau1  = fresh() in
       let tau2 = fresh() in
       let tau1_to_tau2 = mk_fun_ty tau1 tau2 in
       (match gather_exp_ty_substitution
		(ins_env (ins_env gamma f (polyTy_of_monoTy tau1_to_tau2))
			  x (polyTy_of_monoTy tau1))
		e tau2
	with None -> None
	   | Some(pf, sigma) -> 
              let sigma_gamma = env_lift_subst sigma gamma in
	      let sigma_tau1_to_tau2 = monoTy_lift_subst sigma tau1_to_tau2 in
              let delta_env =
                 (ins_env sigma_gamma f (gen sigma_gamma sigma_tau1_to_tau2))
              in 
	      Some(Proof([pf],DecJudgment (gamma, dec, delta_env)),sigma))

(***************** Adapted Infrstructure for MP3, CPS *********************)
let rec freeVarsInExp exp =
    match exp with VarExp x -> [x]
    | ConstExp c -> []
    | MonOpAppExp (m,e) -> freeVarsInExp e
    | BinOpAppExp (b,e1,e2) -> freeVarsInExp e1 @ freeVarsInExp e2
    | IfExp (e1,e2,e3) -> freeVarsInExp e1 @ (freeVarsInExp e2 @ freeVarsInExp e3)
    | AppExp (e1,e2) -> freeVarsInExp e1 @ freeVarsInExp e2
    | FunExp (f,e) -> List.filter (fun y -> not(y = f)) (freeVarsInExp e)
    | LetInExp (x,e1,e2) -> 
       (freeVarsInExp e1) @ (List.filter (fun y -> not(y = x)) (freeVarsInExp e2))
    | LetRecInExp (f,x,e1,e2) ->
      (List.filter (fun y -> not((y = f) || (y = x))) (freeVarsInExp e1)) @
        (List.filter (fun y -> not(y = f)) (freeVarsInExp e2))
    | RaiseExp e -> freeVarsInExp e
    | TryWithExp (e, i, he, hel) ->
      List.fold_left (fun vars (iopt, e) -> ((freeVarsInExp e) @ vars))
        ((freeVarsInExp e) @ (freeVarsInExp he))
        hel

type cont_var = Kvar                      (* _k *)

type cps_cont = 
   External
 | ContVarCPS of cont_var              (* _k *)
 | FnContCPS of string * exp_cps       (* FN x -> exp_cps *)
 | ExnMatch of exn_cont                (* i1 |-> ec1; ... in |-> ecn *)

and exp_cps =
   VarCPS of cps_cont * string
 | ConstCPS of cps_cont * const
 | MonOpAppCPS of cps_cont * mon_op * string * exn_cont
 | BinOpAppCPS of cps_cont * bin_op * string * string * exn_cont
 | IfCPS of string * exp_cps * exp_cps
 | AppCPS of cps_cont * string * string * exn_cont
 | FunCPS of cps_cont * string * cont_var * int * exp_cps
 | FixCPS of cps_cont * string * string * cont_var * int * exp_cps 

and exn_cont =
   ExnContVarCPS of int
 | EmptyExnContCPS
 | UpdateExnContCPS of (int option * exp_cps) list * exn_cont
(* The type of UpdateExnContCPS is really all wrong I think. It should
be basically a closure giving the environement for interpretting
the exp_cps *)

let rec freeVarsInExpCPS cont = (
    match cont with VarCPS (k, x) -> x :: freeVarsInContCPS k
    | ConstCPS (k, c) -> freeVarsInContCPS k
    | MonOpAppCPS (k,m,s,ke) ->
      s :: ((freeVarsInExnContCPS ke) @ (freeVarsInContCPS k))
    | BinOpAppCPS (k,b,r,s,ke) ->
      r :: s :: ((freeVarsInExnContCPS ke) @ (freeVarsInContCPS k))
    | IfCPS (r,e1,e2) -> r :: ((freeVarsInExpCPS e1) @ (freeVarsInExpCPS e2))
    | AppCPS (k,x1,x2,ke) ->
      x1::x2:: ((freeVarsInExnContCPS ke) @ (freeVarsInContCPS k))
    | FunCPS (k,x,Kvar,i,e) ->
      (freeVarsInContCPS k) @
        (List.filter (fun y -> not (x = y)) (freeVarsInExpCPS e))
    | FixCPS (k,f,x,Kvar,i,e) -> (freeVarsInContCPS k) @ 
      (List.filter (fun y -> not ((x = y) || (f = y))) (freeVarsInExpCPS e)) 
    )
and
   freeVarsInContCPS k = (
   match k with External -> []
   | ContVarCPS _ -> []
   | FnContCPS (k, e) -> (freeVarsInExpCPS e)
   | ExnMatch ec -> freeVarsInExnContCPS  ec
 )
and
   freeVarsInExnContCPS ke = (
  match ke with ExnContVarCPS i -> []
    | EmptyExnContCPS -> []
    | UpdateExnContCPS (matching, ke) ->
      List.fold_left (fun vars (iopt, e) -> ((freeVarsInExpCPS e) @ vars))
      (freeVarsInExnContCPS ke)
      matching
    )


(* Fresh Name stuff *)

let int_to_string n =
    let int_to_int_26_list n =
        let rec aux n l =
            if n <= 0 then l else let c = ((n-1) mod 26) in aux ((n -(c+1))/26) (c::l)
        in aux n []
    in
        let rec aux l = match l with [] -> ""
                            | n::ns -> (String.make 1 (Char.chr (n + 97))) ^ aux ns
        in aux (int_to_int_26_list n)

let freshFor lst = 
    let rec fresh_ n = 
        if List.mem (int_to_string n) lst
           then fresh_ (n+1)
        else int_to_string n
    in fresh_ 1

(*  Do I need this for exception continuations?
let (freshIntName , resetIntNameInt) =
    let intNameInt = ref 0 in
    let n() = (let x = !intNameInt in intNameInt := x + 1; string_of_int x) in
    let r() = intNameInt := 0 in
    (n,r)
*)

let (next_index, reset_index) = 
    let count = ref 0 in
    let n() = (let x = !count in count := x + 1; x) in
    let r() = count := 0 in
    (n,r)

(* End Fresh name stuff *)

let rec cps_exp e k ke = 
   match e with 
(*[[x]]k,ke = k x*)
     VarExp x -> (VarCPS (k, x))
(*[[c]]k,ke = k x*)
   | ConstExp c -> (ConstCPS (k, c))
(*[[~ e]]k,ke = [[e]](FN r ke -> k (~ r)), ke, *)
   | MonOpAppExp (m, e) ->
      let r = freshFor (freeVarsInContCPS k)
      in cps_exp e ((FnContCPS (r, MonOpAppCPS (k, m, r, ke)))) ke
(*[[(e1 + e2)]]k,ke = [[e1]](FN r -> [[e2]](FN s -> k (r + s)),ke),ke*)
   | BinOpAppExp (b, e1, e2) ->
      let v2 = freshFor (freeVarsInContCPS k @ freeVarsInExp e1)  in 
      let v1 = freshFor (v2 :: (freeVarsInContCPS k)) in 
      let e2CPS =
       cps_exp e1 ((FnContCPS (v1, BinOpAppCPS(k, b, v1, v2, ke)))) ke in
      cps_exp e2 ((FnContCPS (v2, e2CPS))) ke
(*[[if e1 then e2 else e3]]k,ke = [[e1]]((FN r -> if r then [[e2]]k,ke else [[e3]]k,ke),ke*)
   | IfExp (e1,e2,e3) ->
      let r = freshFor (freeVarsInContCPS k @
                        freeVarsInExp e2 @ freeVarsInExp e3) in 
      let e2cps = cps_exp e2 k ke  in
      let e3cps = cps_exp e3 k ke in 
      cps_exp e1 ((FnContCPS(r, IfCPS(r, e2cps, e3cps)))) ke
(*[[e1 e2]]k,ke = [[e1]](FN r -> [[e2]](FN s -> (r s k ke)),ke),ke*)
   | AppExp (e1,e2) -> 
      let v2 = freshFor (freeVarsInContCPS k @ freeVarsInExp e1) in
      let v1 = freshFor (v2 :: freeVarsInContCPS k) in
      let e1cps = cps_exp e1 (FnContCPS (v1, AppCPS(k, v1, v2, ke))) ke in
      cps_exp e2 (FnContCPS (v2, e1cps)) ke
(*[[fun x -> e]]k,ke = k(FUN x -> fn kx kes-> [[e]]kx,kes) *)
   | FunExp (x,e) ->
     let i = next_index() in
     let ecps = cps_exp e (ContVarCPS Kvar) (ExnContVarCPS i) in
     FunCPS (k, x, Kvar, i, ecps)
(*[[let x = e1 in e2)]]k,ke = [[e1]](FN x -> [[e2]]k,ke),ke *)
   | LetInExp (x,e1,e2) -> 
     let e2cps = cps_exp e2 k ke in 
     let fnk = FnContCPS(x,e2cps) in cps_exp e1 fnk ke
(*[[let rec f x = e1 in e2]]k,ke =
  (FN f -> [[e2]]k,ke)(FIX f. FUN x -> fn k',ke' => [[e1]]k',ke')*)
   | LetRecInExp(f,x,e1,e2) ->
     let i = next_index() in
     let e1cps = cps_exp e1 (ContVarCPS Kvar) (ExnContVarCPS i)  in 
     let e2cps = cps_exp e2 k ke in 
     let fnk = FnContCPS(f,e2cps) in
     FixCPS(fnk,f,x,Kvar,i,e1cps)
(* [[try e with n0 -> e0 | ... | nm -> em]]k,ke =
   [[e]]k, [(n0 |-> [[e0]]k,ke); ... (nm |-> [[em]]k,ke)] + ke *)
   | TryWithExp(e, n0, e0, rem_match) ->
     let match_cps =
         List.fold_right
          (fun (n,en) -> fun exn_match ->
           let ecps = cps_exp en k ke in
               (n, ecps)::exn_match)
          ((n0, e0):: rem_match)
          []
     in cps_exp e k (UpdateExnContCPS(match_cps, ke))

(* [[raise e]]k,ke = [[e]](FN r -> match_exn r with ke), ke *)
   | RaiseExp e -> cps_exp e ((ExnMatch ke)) ke


(********************* From MP6, shared with MP7 ************************)
(*                  values                   *)

	
type value =
    UnitVal                             
  | BoolVal of bool
  | IntVal of int                       
  | FloatVal of float
  | StringVal of string                 
  | PairVal of value * value
  | ListVal of value list 
  | CPSClosureVal
      of string * cont_var *  int * exp_cps *  (cps_env_entry list)
  | CPSRecClosureVal
      of string * string * cont_var * int * exp_cps * (cps_env_entry list)

and cps_env_entry =
     ValueBinding of (string * value)
   | ContBinding of (cont_var * (cps_cont * (cps_env_entry list)))
   | ExnContBinding of (int * (exn_cont * cps_env_entry list))

let rec lookup_value env x =
   match env with [] -> None
   | (bnd:: rem_env) ->
    (match bnd with ValueBinding(y,v) ->
      if x = y then Some v else lookup_value rem_env x
     | _ -> lookup_value rem_env x)

let rec lookup_cont env k=
   match env with [] -> None
   | (bnd:: rem_env) ->
    (match bnd with ContBinding(k', c) -> Some c
     | _ -> lookup_cont rem_env k)

let rec lookup_exn_cont env x =
   match env with [] -> None
   | (bnd:: rem_env) ->
    (match bnd with ExnContBinding(y,ec) ->
      if x = y then Some ec else lookup_exn_cont rem_env x
     | _ -> lookup_exn_cont rem_env x)

let rec string_of_value v =
   match v with
    UnitVal           -> "()"
  | IntVal n          -> if n < 0 then "~"^string_of_int(abs n) else string_of_int n 
  | FloatVal r        -> string_of_float r
  | BoolVal true      -> "true"
  | BoolVal false     -> "false"
  | StringVal s       ->  ("\"" ^  (String.escaped s) ^ "\"")
  | PairVal (v1,v2)   -> "("^
                         string_of_value v1^  ", "^
                         string_of_value v2^
                          ")"
  | ListVal l         -> "["^
                         (let rec pl = function
                              []     -> "]"
                              | [v1] -> string_of_value v1 ^ "]"
                              | v1 :: vl -> string_of_value v1 ^ "; " ^ pl vl
                              in pl l)
  | CPSClosureVal (x, Kvar, i, e, m) ->  ("<some closure>")
  | CPSRecClosureVal (f, x, Kvar, i, e, m)  ->  ("<some recvar>")

let print_value v = print_string (string_of_value v)


(* Basic Evaulation Support *)

let const_to_val c = 
  match c with
    BoolConst b   -> BoolVal b
  | IntConst i    -> IntVal i
  | FloatConst f  -> FloatVal f
  | StringConst s -> StringVal s
  | NilConst      -> ListVal []
  | UnitConst     -> UnitVal

type ord = GT (* greater *) | EQ (* equal *) | LT (* less *)

let gencompare a b =
  if a < b then LT
  else if a > b then GT
  else EQ

let rec compareVal v1 v2 =
   match (v1, v2) with
    (UnitVal, UnitVal) -> EQ
  | (BoolVal b1, BoolVal b2) -> gencompare b1 b2
  | (IntVal n1, IntVal n2) -> gencompare n1 n2
  | (FloatVal r1, FloatVal r2) -> gencompare r1 r2
  | (StringVal s1, StringVal s2) -> gencompare s1 s2
  | (PairVal (u1,u2), PairVal (v1,v2)) -> paircompare (u1,u2) (v1,v2)
  | (ListVal l1, ListVal l2) -> listcompare l1 l2
  | (CPSClosureVal _, _) -> raise (Failure "Cannot compare funtions")
  | (_, CPSClosureVal _) -> raise (Failure "Cannot compare funtions")
  | (CPSRecClosureVal _, _) -> raise (Failure "Cannot compare funtions")
  | (_, CPSRecClosureVal _) -> raise (Failure "Cannot compare funtions")
  | (_, _) -> raise (Failure "Type checking error")

and paircompare (u1, u2) (v1, v2) =
     (match compareVal u1 v1 with EQ -> compareVal u2 v2
      | r -> r)

and listcompare l1 l2 =
   match l1 with
     [] -> (match l2 with [] -> EQ | _ -> LT)
   | (x::xs) -> (match l2 with (y::ys) ->
                   (match compareVal x y with EQ -> listcompare xs ys
                     | r -> r)
                  | [] -> GT)

type val_or_exn = Value of value | Exn of int

let binOpApply binop v1 v2 =
  match (binop, v1, v2) with
    (IntPlusOp, IntVal n1, IntVal n2)      -> Value(IntVal (n1 + n2))
  | (IntMinusOp, IntVal n1, IntVal n2)     -> Value(IntVal (n1 - n2))
  | (IntTimesOp, IntVal n1, IntVal n2)     -> Value(IntVal (n1 * n2))
  | (IntDivOp, IntVal n1, IntVal n2)       -> 
    if n2 = 0 then Exn 0 (* Part 2, Problem 16 *) else Value(IntVal (n1 / n2))
  | (FloatPlusOp, FloatVal n1, FloatVal n2)   -> Value(FloatVal (n1 +. n2))
  | (FloatMinusOp, FloatVal n1, FloatVal n2)  -> Value(FloatVal (n1 -. n2))
  | (FloatTimesOp, FloatVal n1, FloatVal n2)  -> Value(FloatVal (n1 *. n2))
  | (FloatDivOp, FloatVal n1, FloatVal n2)    ->
    if n2 = 0.0 then Exn 0 (* Part 2, Problem 16 *) else Value(FloatVal (n1 /. n2))
  | (ConcatOp, StringVal s1, StringVal s2) -> Value(StringVal (s1 ^ s2))
  | (ConsOp, v, ListVal l)                 -> Value(ListVal (v::l))
  | (CommaOp, u1, u2)                      -> Value(PairVal (u1,u2))
  | (EqOp, u1, u2)                         -> Value(BoolVal (match compareVal u1 u2 with EQ -> true
                                                        | _ -> false))
  | (GreaterOp, u1, u2)                    -> Value(BoolVal (match compareVal u1 u2 with GT -> true
                                                        | _ -> false))
  | (_, _, _)                              -> raise (Failure "typechecking error")

let monOpApply op v =
  match (op, v) with
  | (IntNegOp, IntVal n)         -> Value(IntVal (- n))
 (* Part 2, Problem 16 *)
  | (HdOp, ListVal [])         -> Exn 0
  | (HdOp, ListVal (v::vs))    -> Value(v)
 (* Part 2, Problem 16 *)
  | (TlOp, ListVal [])         -> Exn 0
  | (TlOp, ListVal (v::vs))    -> Value(ListVal vs)
  | (FstOp, PairVal (v1, v2))  -> Value(v1)
  | (SndOp, PairVal (v1, v2))  -> Value(v2)
  | (PrintStringOp, StringVal s) -> (print_string s; Value(UnitVal))
  | (_ , _ )                   -> failwith "monOpApply: bad input"

(*-----------------------Type for CPS Transition Semantics results---------------*)

type step_result =
   Intermediate of (cps_env_entry list * exp_cps)
 | Final of value
 | UncaughtException of int
 | Failed


(* ----------------------------app_exn_handler_cont ------------------------------*)

let rec match_lookup l n =
    match l with [] -> None
    | (Some m, v)::more -> if n = m then Some v else match_lookup more n
    | (None, v)::_ -> Some v

let rec app_exn_handler_cont env ce n =
    match ce with EmptyExnContCPS -> UncaughtException n
    |  ExnContVarCPS i ->
      (match lookup_exn_cont env i with None -> Failed
       | Some (ke',env') -> app_exn_handler_cont env' ke' n)
    | UpdateExnContCPS (handler, ce') -> 
      (match match_lookup handler n with
         None -> app_exn_handler_cont env ce' n
       | Some ecps -> Intermediate(env, ecps))
